define('bitbucket/internal/feature/file-content/commit-selector', [
    'aui',
    'jquery',
    'lodash',
    'bitbucket/util/navbuilder',
    'bitbucket/internal/model/path',
    'bitbucket/internal/util/ajax',
    'bitbucket/internal/util/events',
    'bitbucket/internal/widget/paged-scrollable'
], function(
    AJS,
    $,
    _,
    nav,
    Path,
    ajax,
    events,
    PagedScrollable
) {

    function CommitSelector(buttonSelector, id) {
        var $selectorButton = this._$selectorButton = $(buttonSelector);
        this._id = id;
        this._scrollPaneSelector = '#inline-dialog-' + id + ' .commit-selector';
        this._listSelector = this._scrollPaneSelector + " > ul";

        var self = this;
        var hideOnEscapeKeyUp = function(e) {
            if(e.keyCode === AJS.keyCode.ESCAPE) {
                self.hide();
                e.preventDefault();
            }
        };
        var showOnClick = function(e) {
            if (self.isButtonEnabled()) {
                self.show();
            }
            e.preventDefault();
        };
        var hideOnClick = function(e) {
            self.hide();
            e.preventDefault();
        };
        // _itemClicked is used in file-history-test.js
        var itemClicked = this._itemClicked = function(e) {
            self.hide();

            var $li = $(this);
            var $a = $li.children('a');
            var commitId = $a.attr('data-id');
            var commit = self._visibleCommits[commitId];
            var path = null;
            var sincePath = null;


            if (commit.properties && commit.properties.change) {
                path = commit.properties.change.path ? new Path(commit.properties.change.path) : null;
                sincePath = commit.properties.change.srcPath ? new Path(commit.properties.change.srcPath) : null;
            }

            events.trigger('bitbucket.internal.feature.commitselector.commitSelected', self, commit, self._pullRequest, path, sincePath);
            e.preventDefault();
        };
        var scrollable = null;
        var _dialogInitialised = false;
        var $currentContent;

        var onShowDialog = function ($content, trigger, showPopup) {
            if (!_dialogInitialised) {
                _dialogInitialised = true;

                $currentContent = $content.html(bitbucket.internal.feature.fileContent.commitSelector());
                $currentContent.on('click', 'li.commit-list-item', itemClicked);

                showPopup();
                scrollable = self._createScrollable();
                self._visibleCommits = {};
                setTimeout(function() {
                    $content.find('.spinner-container').spin();
                    scrollable.init();
                }, 0);
            } else {
                showPopup();
                _.defer(function () {
                    $(self._listSelector).find(':first a').focus();
                });
            }

            $selectorButton.off('click', showOnClick);
            $selectorButton.on('click', hideOnClick);
            $(document).on('keyup', hideOnEscapeKeyUp);
        };

        var onHideDialog = function () {
            $(document).off('keyup', hideOnEscapeKeyUp);
            $selectorButton.off('click', hideOnClick);
            $selectorButton.on('click', showOnClick);
            if ($(document.activeElement).closest(self._scrollPaneSelector).length) {
                // if the focus is inside the dialog, you get stuck when it closes.
                document.activeElement.blur();
            }
        };

        this.resetDialog = function () {
            self.hide();
            scrollable && scrollable.reset();
            $(document).off('keyup', hideOnEscapeKeyUp);
            $currentContent && $currentContent.off('click', 'li.commit-list-item', itemClicked);
            _dialogInitialised = false;
        };

        this._inlineDialog = AJS.InlineDialog($selectorButton, id, onShowDialog, {
            hideDelay: null,
            width : 483,
            noBind: true,
            hideCallback: onHideDialog
        });

        $selectorButton.on('click', showOnClick);
        this._events = events.chain()
            .on('bitbucket.internal.feature.commitselector.commitSelected', this.resetDialog)
            .on('bitbucket.internal.page.*.revisionRefChanged', this.resetDialog)
            .on('bitbucket.internal.page.*.pathChanged', this.resetDialog);
    }

    CommitSelector.prototype.init = function(options) {
        this._followRenames = options.followRenames;
        this._headRevisionRef = options.headRevisionReference;
        this._itemTemplate = options.itemTemplate;
        this._itemUrl = options.itemUrl;
        this._mode = options.mode;
        this._path = options.path;
        this._pullRequest = options.pullRequest;
        this._selectedRevisionRef = options.selectedRevisionReference;
    };

    CommitSelector.prototype.destroy = function () {
        this.resetDialog();
        this._events.destroy();
        this._inlineDialog.remove();
    };

    CommitSelector.prototype.show = function() {
        this._inlineDialog.show();
        $(this._listSelector).find(':first a').focus();
    };

    CommitSelector.prototype.hide = function() {
        this._inlineDialog.hide();
    };

    CommitSelector.prototype._createScrollable = function() {
        var scrollable = new PagedScrollable(this._scrollPaneSelector, {
            bufferPixels : 0,
            pageSize : 25,
            paginationContext: 'file-history'
        });
        scrollable.requestData = $.proxy(this.requestData, this);
        scrollable.attachNewContent = $.proxy(this.attachNewContent, this);

        var oldOnFirstDataLoaded = scrollable.onFirstDataLoaded;
        var self = this;
        scrollable.onFirstDataLoaded = function() {
            $(self._listSelector).find(':first a').focus();
            return oldOnFirstDataLoaded.apply(this, arguments);
        };

        return scrollable;
    };

    CommitSelector.prototype.requestData = function(start, limit) {
        this._inlineDialog.find('.spinner-container').spin();
        var urlBuilder;

        // Could potentially be extracted out of the commit selector as it toggles between file history and
        // pull request commit selector
        if (this._pullRequest) {
            urlBuilder = nav.rest()
                .currentRepo()
                .pullRequest(this._pullRequest.id)
                .commits()
                .withParams({
                    start : start, limit : limit,
                    avatarSize : bitbucket.internal.widget.avatarSizeInPx({ size: 'xsmall' })
                });
        } else {
            urlBuilder = nav.rest()
                .currentRepo()
                .commits()
                .withParams({
                    followRenames: this._followRenames,
                    path: this._path.toString(),
                    until: this._headRevisionRef.getId(),
                    start: start, limit: limit,
                    avatarSize: bitbucket.internal.widget.avatarSizeInPx({size: 'xsmall'})
                });
        }

        return ajax.rest({
            url: urlBuilder.build()
        }).done(function(data) {
            if (data.size > 0) {
                return data;
            } else {
                // BSERV-8673 If there is no data the file must have been created in a merge
                // so do the REST call again with followRenames: false so merge commits are included
                return ajax.rest({
                    url: urlBuilder.withParams({
                        followRenames: false
                    }).build()
                });
            }
        });
    };

    function addCommitsToMap(commits, cache) {
        _.forEach(commits, function(commit) {
            cache[commit.id] = commit;
        });
    }

    CommitSelector.prototype.attachNewContent = function(data) {
        addCommitsToMap(data.values, this._visibleCommits);
        var self = this;
        var commitSelectorItems = data.values.map(function(commit) {
            return bitbucket.internal.widget.commit.commitListItem({
                commit: commit,
                isSelected: (self._selectedRevisionRef && self._selectedRevisionRef.getId()) === commit.id,
                content: self._itemTemplate({
                    commit: commit,
                    href: self._itemUrl(commit, self)
                })
            });
        });

        $(this._listSelector)
            .append(commitSelectorItems)
            .append(bitbucket.internal.feature.fileContent.noMoreResults ({
                isLastPage: data.isLastPage,
                lastPageMessage: AJS.I18n.getText('bitbucket.web.file.history.allhistoryfetched')
            }));
        $(this._scrollPaneSelector).children('.spinner-container').spinStop();
        if (data.isLastPage) {
            $(this._scrollPaneSelector).children('.spinner-container').remove();
        }
    };

    CommitSelector.prototype.isButtonEnabled = function() {
        return !this._$selectorButton.prop('disabled');
    };

    return CommitSelector;
});
