define(
    'bitbucket/internal/bbui/time',
    ['exports', 'module', 'aui', 'lodash', 'moment'],
    function (exports, module, _aui, _lodash, _moment) {
        'use strict';

        function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

        function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError('Cannot call a class as a function'); } }

        var _AJS = _interopRequireDefault(_aui);

        var _2 = _interopRequireDefault(_lodash);

        var _moment2 = _interopRequireDefault(_moment);

        var hasOwn = Object.prototype.hasOwnProperty;
        var dateFormatCache = {};
        var dateTokenizer = /d{1,2}|'[^']+'|M{1,4}|y{2,4}|h{1,2}|H{1,2}|m{2}|s{2}|S{1,4}|Z{1,2}|z{1,2}|a|:|-|\/|\s+/g;

        /**
         * Exposes a `types` object which is an enum of time and date types.
         * @class
         */

        var Type = function Type(key, isAge) {
            _classCallCheck(this, Type);

            this.key = key;
            this.isAge = isAge;
        };

        Type.types = {};

        ['shortAge', 'longAge', 'short', 'long', 'full', 'timestamp'].forEach(function (t) {
            Type.types[t] = new Type(t, t.toLowerCase().indexOf('age') !== -1);
        });

        // The `dateFormats` object is copied from the `date-format.properties` file in Bitbucket Server and should be kept in
        // sync with the Java implementation.
        var dateFormats = {
            'time.date.format.short': 'dd MMM yyyy',
            'time.date.format.long': 'dd MMMM yyyy',
            'time.date.format.full': 'dd MMMM yyyy hh:mm a',
            'time.date.format.datetime': 'dd MMM - hh:mm a',
            'time.date.format.timestamp': 'yyyy-MM-dd\'T\'HH:mm:ssZ',
            'time.date.format.timeonly': 'hh:mm a'
        };

        function getTextForRelativeAge(age, type, param, customMapping) {
            if (customMapping) {
                return getTextForCustomAge(age, param, customMapping);
            }

            return type === Type.types.shortAge ? getTextForShortAge(age, param) : getTextForLongAge(age, param);
        }

        function getTextForShortAge(age, param, mapping) {
            // eslint-disable-line no-unused-vars
            // NOTE: AJS cannot be an AMD dependency as the minifier then changes the AJS.I18n.getText references
            // NOTE: and the transformer doesn't do any translation. IMO this is a webresources _bug_ (https://ecosystem.atlassian.net/browse/PLUGWEB-17).
            switch (age) {
                case 'aMomentAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.a.moment.ago');
                case 'oneMinuteAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.one.minute.ago');
                case 'xMinutesAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.x.minutes.ago', param);
                case 'oneHourAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.one.hour.ago');
                case 'xHoursAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.x.hours.ago', param);
                case 'oneDayAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.one.day.ago');
                case 'xDaysAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.x.days.ago', param);
                case 'oneWeekAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.short.one.week.ago');
                default:
                    return null;
            }
        }

        function getTextForLongAge(age, param) {
            // NOTE: AJS cannot be an AMD dependency as the minifier then changes the AJS.I18n.getText references
            // NOTE: and the transformer doesn't do any translation. IMO this is a webresources _bug_ (https://ecosystem.atlassian.net/browse/PLUGWEB-17).
            switch (age) {
                case 'aMomentAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.a.moment.ago');
                case 'oneMinuteAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.one.minute.ago');
                case 'xMinutesAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.x.minutes.ago', param);
                case 'oneHourAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.one.hour.ago');
                case 'xHoursAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.x.hours.ago', param);
                case 'oneDayAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.one.day.ago');
                case 'xDaysAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.x.days.ago', param);
                case 'oneWeekAgo':
                    return _AJS['default'].I18n.getText('bitbucket.component.time.format.long.one.week.ago');
                default:
                    return null;
            }
        }

        function getTextForCustomAge(age, param, customMapping) {
            if (customMapping) {
                switch (age) {
                    case 'aMomentAgo':
                        return customMapping.aMomentAgo(param);
                    case 'oneMinuteAgo':
                        return customMapping.oneMinuteAgo(param);
                    case 'xMinutesAgo':
                        return customMapping.xMinutesAgo(param);
                    case 'oneHourAgo':
                        return customMapping.oneHourAgo(param);
                    case 'xHoursAgo':
                        return customMapping.xHoursAgo(param);
                    case 'oneDayAgo':
                        return customMapping.oneDayAgo(param);
                    case 'xDaysAgo':
                        return customMapping.xDaysAgo(param);
                    case 'oneWeekAgo':
                        return customMapping.oneWeekAgo(param);
                    default:
                        return null;
                }
            }
        }

        function toMomentFormat(javaDateFormat) {

            /*jshint boss:true */
            if (hasOwn.call(dateFormatCache, javaDateFormat)) {
                return dateFormatCache[javaDateFormat];
            }
            var momentDateFormat = "";
            var token = undefined;
            dateTokenizer.exec('');
            while (token = dateTokenizer.exec(javaDateFormat)) {
                token = token[0];
                switch (token.charAt(0)) {
                    case "'":
                        momentDateFormat += '[' + token.substring(1, token.length - 1) + ']';
                        break;
                    case 'd':
                    // falls through
                    case 'y':
                    // falls through
                    case 'a':
                        momentDateFormat += token.toUpperCase();
                        break;
                    default:
                        momentDateFormat += token;
                }
            }
            dateFormatCache[javaDateFormat] = momentDateFormat;
            return momentDateFormat;
        }

        function getFormatString(type) {
            switch (type.key) {
                case 'short':
                case 'shortAge':
                    return dateFormats['time.date.format.short'];
                case 'long':
                case 'longAge':
                    return dateFormats['time.date.format.long'];
                case 'full':
                    return dateFormats['time.date.format.full'];
                case 'timestamp':
                    return dateFormats['time.date.format.timestamp'];
                default:
                    return null;
            }
        }

        // Server timezone offset for Bitbucket Server, otherwise defaults to get the browser timezone offset
        function getTimezoneOffset() {
            var contentElement = document.getElementById("content");

            if (contentElement) {
                var timezone = parseInt(contentElement.getAttribute('data-timezone'), 10);
                if (!isNaN(timezone)) {
                    return timezone;
                }
            }

            return new Date().getTimezoneOffset();
        }

        function getFormattedTimezoneOffset(hourMinuteSeparator, optOffset) {
            var offset = typeof optOffset === 'number' ? optOffset : _exports.getTimezoneOffset(); // eslint-disable-line no-use-before-define
            var abs = Math.abs(offset);
            var hour = Math.floor(abs / 60); // eslint-disable-line no-magic-numbers
            var minute = abs % 60; // eslint-disable-line no-magic-numbers
            var ret = '';

            ret += offset <= 0 ? '+' : '-'; // flip the sign
            ret += _2['default'].padLeft(hour.toString(), 2, '0');
            ret += hourMinuteSeparator || '';
            ret += _2['default'].padLeft(minute.toString(), 2, '0');
            return ret;
        }

        function localiseTimezone(date, optOffset) {
            var converted = date.clone();
            var offset = typeof optOffset === 'number' ? optOffset : _exports.getTimezoneOffset(); // eslint-disable-line no-use-before-define
            if (-date.utcOffset() !== offset) {
                // set the time correctly for the new timezone
                converted.add(-date.utcOffset() - offset, 'm');
            }
            return converted;
        }

        function isYesterday(now, date) {
            var end = now.clone().add(1, 'd').hours(0).minutes(0).seconds(0).milliseconds(0).subtract(-date.utcOffset() - _exports.getTimezoneOffset(), 'm'); // eslint-disable-line no-use-before-define
            while (end > now) {
                end.subtract(1, 'd');
            }
            var start = end.clone().subtract(1, 'd');
            return start <= date && date < end;
        }

        function getMinutesBetween(start, end) {
            return Math.floor(end.diff(start, 'minutes', true));
        }

        function getHoursBetween(start, end) {
            var hourDiff = end.diff(start, 'hours', true); // Moment's diff does a floor rather than a round so we pass 'true' for a float value
            return Math.round(hourDiff); // Then round it ourself
        }

        function getDaysBetween(start, end) {
            return Math.floor(end.diff(start, 'days', true));
        }

        function formatDate(momentDate, type, customMapping) {
            if (momentDate && type) {
                if (type.isAge) {
                    return formatDateWithRelativeAge(momentDate, type, null, customMapping);
                }
                return formatDateWithFormatString(momentDate, type);
            }
            return null;
        }

        /**
         * Formats the input date using a named `type`, with an optional offset for timezone handling
         * @param {MomentDate} date - The date to be formatted
         * @param {String} type - either 'shortAge', 'longAge', 'short', 'long', 'full' or 'timestamp'
         * @param {Number} optOffset - the timezone offset
         * @returns {String}
         */
        function formatDateWithFormatString(date, type, optOffset) {
            var offset = typeof optOffset === 'number' ? optOffset : _exports.getTimezoneOffset(); // eslint-disable-line no-use-before-define

            var localisedDate = localiseTimezone(date, offset);

            //We need to replace timezones with the timezone from exports.getTimezoneOffset(), which moment can't do.
            var formatString = toMomentFormat(getFormatString(type)).replace(/Z+/g, function (input) {
                //intentional simplification: treat three or more Zs as ZZ.
                return '[' + getFormattedTimezoneOffset(input.length === 1 ? '' : ':', offset) + ']';
            });

            return localisedDate.format(formatString);
        }

        /**
         * Compares the input date to the current date and returns a relative time
         * @param {MomentDate} date - The date to compare against
         * @param {String} type - either 'shortAge', 'longAge', 'short', 'long', 'full' or 'timestamp'
         * @param {MomentDate} now - the current date, or creates a new moment() instance with the current date.
         * @param {Object} customMapping - and object with methods that override the standard relative age i18n strings
         * @returns {String}
         */
        function formatDateWithRelativeAge(date, type, now, customMapping) {
            now = now || (0, _moment2['default'])();

            /*eslint-disable no-magic-numbers*/
            if (date <= now) {
                if (date > now.clone().subtract(1, 'm')) {
                    return getTextForRelativeAge('aMomentAgo', type, null, customMapping);
                } else if (date > now.clone().subtract(2, 'm')) {
                    return getTextForRelativeAge('oneMinuteAgo', type, null, customMapping);
                } else if (date > now.clone().subtract(50, 'm')) {
                    return getTextForRelativeAge('xMinutesAgo', type, getMinutesBetween(date, now), customMapping);
                } else if (date > now.clone().subtract(90, 'm')) {
                    return getTextForRelativeAge('oneHourAgo', type, null, customMapping);
                } else if (isYesterday(now, date) && date < now.clone().subtract(5, 'h')) {
                    return getTextForRelativeAge('oneDayAgo', type, null, customMapping);
                } else if (date > now.clone().subtract(1, 'd')) {
                    return getTextForRelativeAge('xHoursAgo', type, getHoursBetween(date, now), customMapping);
                } else if (date > now.clone().subtract(7, 'd')) {
                    // if it's not yesterday then don't say it's one day ago
                    return getTextForRelativeAge('xDaysAgo', type, Math.max(getDaysBetween(date, now), 2), customMapping);
                } else if (date > now.clone().subtract(8, 'd')) {
                    return getTextForRelativeAge('oneWeekAgo', type, null, customMapping);
                } else if (customMapping && typeof customMapping.defaultType === 'function') {
                    return customMapping.defaultType(formatDateWithFormatString(date, type));
                }
            }

            /*eslint-enable no-magic-numbers*/

            return formatDateWithFormatString(date, type);
        }

        /**
         * Converts a date into a specified format
         * @param {Date|Number|String} dateOrNumberOrString - the date to format. Supports all formats in {@link http://momentjs.com/docs/#/parsing/|moment.js}
         * @param {String} typeString - either 'shortAge', 'longAge', 'short', 'long', 'full' or 'timestamp'
         * @param {Object} customMapping - and object with methods that override the standard relative age i18n strings
         * @returns {String}
         */
        function format(dateOrNumberOrString, typeString, customMapping) {
            return formatDate(dateOrNumberOrString ? (0, _moment2['default'])(dateOrNumberOrString) : null, Type.types[typeString], customMapping);
        }

        // the getTimezoneOffset() method is expected to be on the exports object, and there's a default implementation
        // in this component that uses the browser time, but Server we override it and set it to the server's timezone
        var _exports = {
            formatDateWithFormatString: formatDateWithFormatString,
            formatDateWithRelativeAge: formatDateWithRelativeAge,
            format: format,
            getTimezoneOffset: getTimezoneOffset,
            FormatType: Type
        };

        module.exports = _exports;
    }
);