define('bitbucket/internal/page/pull-requests-list', [
    'jquery',
    'lodash',
    'bitbucket/util/events',
    'bitbucket/util/navbuilder',
    'bitbucket/util/state',
    'bitbucket/internal/bbui/models',
    'bitbucket/internal/bbui/pull-request-list',
    'bitbucket/internal/feature/pull-request/list/analytics',
    'bitbucket/internal/model-transformer',
    'bitbucket/internal/util/history',
    'bitbucket/internal/util/shortcuts',
    'exports'
], function (
    $,
    _,
    events,
    nav,
    state,
    models,
    PullRequestList,
    listAnalytics,
    transformer,
    history,
    shortcuts,
    exports
) {
    'use strict';

    // The filter url map is used to convert filter params to url params
    // and vice versa (maps url params back to filter params)
    var filterUrlMap = {
        author_id: 'author',
        reviewer_self: 'reviewing',
        state: 'state',
        target_ref_id: 'at'
    };

    /**
     * Add a set of params to a given URL
     * @param {string} url
     * @param {Object<string, string>} params
     * @returns {string}
     */
    function urlWithParams (url, params) {
        var uri = nav.parse(url);
        Object.keys(params).forEach(function (k) {
            if (params[k] != null && params[k] !== false) {
                uri.replaceQueryParam(k, params[k]);
            } else {
                // delete params that are empty
                uri.deleteQueryParam(k);
            }
        });
        return uri.toString();
    }

    /**
     * Get a filter object from the URL params
     *
     * @returns {Object<string, string>}
     */
    function filterFromUrlParams () {
        var params = {};
        var urlFilterMap = _.invert(filterUrlMap);
        var uri = nav.parse(window.location);

        // get the query params for each of the items in the filter map
        _.values(filterUrlMap).forEach(function (val) {
            if (uri.getQueryParamValue(val)) {
                params[val] = uri.getQueryParamValue(val);
            }
        });

        return _.mapKeys(params, function (v, k) {
            return urlFilterMap[k] || k;
        });
    }

    /**
     * Update the URL with the appropriate params based on given filter
     *
     * @param {Object} filter
     */
    function updateUrlWithFilterParams (filter) {
        var mappedFilterParams = _.mapKeys(filter, function (v, k) {
            return filterUrlMap[k] || k;
        });
        history.replaceState(mappedFilterParams, null, urlWithParams(window.location, mappedFilterParams));
    }

    /**
     *
     * @param {Object} opts
     * @param {Page<PullRequestJSON>} initialData - a page of PRs
     * @param {StashUserJSON} selectedAuthor - author who is selected in the filter
     * @param {RefJSON?} selectedTargetBranch - target branch which is selected in the filter
     */
    exports.onReady = function (opts) {
        events.on('bitbucket.internal.widget.keyboard-shortcuts.register-contexts', function (keyboardShortcuts) {
            keyboardShortcuts.enableContext('pull-request-list');
        });
        var filterParams = filterFromUrlParams();
        var repo = transformer.repository(state.getRepository());
        var prList = new PullRequestList($('#pull-requests-content'), {
            initialData: opts.initialData,
            repository: repo,
            currentUser: transformer.user(state.getCurrentUser()),
            selectedAuthor: transformer.user(opts.selectedAuthor),
            selectedTargetBranch: transformer.ref(opts.selectedTargetBranch, 'branch', repo),
            filter: filterParams, // use the inverse of the filterMap
            authorFilter: {
                role: models.UserRole.AUTHOR.toUpperCase()
            },
            gettingStarted: opts.gettingStarted
        });
        prList.on('state-change', function (state) {
            updateUrlWithFilterParams(state);
            listAnalytics.onFilterChanged(state);
        });

        prList.pullRequestTable.on('pagination', listAnalytics.onPaginate);
        shortcuts.bind('requestMoveToNextHandler', prList.getKeyboardAction('moveNext'));
        shortcuts.bind('requestMoveToPreviousHandler', prList.getKeyboardAction('movePrevious'));
        shortcuts.bind('requestOpenItemHandler', prList.getKeyboardAction('openItem'));

        listAnalytics.init({
            filterParams: filterParams
        });
    };
});
