define('bitbucket/internal/feature/repository/repository-search', [
    'aui',
    'jquery',
    'lodash',
    'xregexp',
    'bitbucket/util/navbuilder',
    'bitbucket/internal/util/events',
    'bitbucket/internal/util/shortcuts',
    'bitbucket/internal/widget/searchable-selector'
], function(
    AJS,
    $,
    _,
    XRegExp,
    nav,
    events,
    shortcuts,
    SearchableSelector
) {
    // Get a hold of the list of recent repositories when they're loaded
    var recentRepositories = null;
    events.on('bitbucket.internal.feature.repositories.recent.loaded', function(data) {
        recentRepositories = data;
    });

    /**
     * @return {RepositorySearch} The new RepositorySearch instance
     *
     * @param {HTMLElement|jQuery}  searchField     The text input for the search
     * @param {HTMLElement|jQuery}  trigger         The element off which the dialog should be positioned. Can be the same element as searchField
     * @param {Object}              options         A hash of options, valid options are specified in SearchableSelector.prototype.defaults
     */
    var RepositorySearch = function(searchField, trigger, options) {
        options = options || {};
        options.externalSearchField = searchField;

        var repositorySearch = this.init(trigger, options);
        repositorySearch._bindSearchFieldEvents();
        repositorySearch.dialog.on('click', 'a', function() {
            events.trigger('bitbucket.internal.ui.nav.search.result.clicked', null, { repositoryId: $(this).attr('data-repo-id') });
        });

        shortcuts.bind('focusRepositorySearch', _.ary($.fn.focus.bind($(searchField)), 0));

        return repositorySearch;
    };

    $.extend(RepositorySearch.prototype, SearchableSelector.prototype);

    RepositorySearch.prototype._generatePreloadData = function() {
        // Don't show anything yet if we don't have any recent repo matches to show
        if (recentRepositories == null) {
            return null;
        }
        // Filter the recent repositories list as the server-side would
        var values;
        var searchRequest = this._parseSearchTerm(this._getSearchTerm());
        if (searchRequest) {
            var projectRegex = searchRequest.projectName && new RegExp('^' + XRegExp.escape(searchRequest.projectName), 'i');
            var repositoryRegex = searchRequest.repositoryName && new RegExp('^' + XRegExp.escape(searchRequest.repositoryName), 'i');
            values = _.filter(recentRepositories.values, function(value) {
                return (searchRequest.projectName == null || value.project.name.match(projectRegex) != null) &&
                    (searchRequest.repositoryName == null || value.name.match(repositoryRegex) != null);
            });
        } else {
            values = recentRepositories.values;
        }
        // Sort the results as the server-side would
        values.sort(function(a, b) {
            return (searchRequest && searchRequest.projectName ?  a.project.name.localeCompare(b.project.name) : 0) ||
                a.name.localeCompare(b.name) ||
                a.id - b.id;
        });
        return _.extend({}, recentRepositories, {
            values: values,
            size: values.length,
            isLastPage: false
        });
    };

    RepositorySearch.prototype.defaults = $.extend({}, SearchableSelector.prototype.defaults, {
        extraClasses: 'repository-search-dialog',
        url: nav.rest().allRepos().withParams({
            avatarSize: bitbucket.internal.widget.avatarSizeInPx({ size: 'xsmall' })
        }).build(),
        queryParamsBuilder: function(searchTerm, start, limit) {
            var searchRequest = this._parseSearchTerm(searchTerm);
            if (searchRequest == null) {
                return null;
            }

            return {
                start: start,
                limit: limit,
                name: searchRequest.repositoryName,
                projectname: searchRequest.projectName
            };
        },
        dataTransform: function(data, searchTerm) {
            var searchRequest = this._parseSearchTerm(searchTerm);
            if (searchRequest == null) {
                return data;
            }

            return $.extend({
                repositorySearchTerm: searchRequest.repositoryName,
                projectSearchTerm: searchRequest.projectName
            }, data);
        },
        preloadData: RepositorySearch.prototype._generatePreloadData,
        alwaysShowPreload: true,
        followLinks: true,
        resultsTemplate: bitbucket.internal.feature.repository.repositorySearchResults,
        noResultsText: AJS.I18n.getText('bitbucket.web.repository.search.results.empty'),
        noMoreResultsText: AJS.I18n.getText('bitbucket.web.repository.search.results.end'),
        adjacentItems: false,
        clearResultsOnSearch: false,
        popUpOffsetY: 1,
        paginationContext: 'repository-search'
    });

    RepositorySearch.prototype._bindSearchFieldEvents = function() {
        var repositorySearch = this;
        // Prevent form submission must be bound on keydown
        this.$searchField.on('keydown.repository-search', function(e) {
            if (e.which === AJS.keyCode.ENTER) {
                e.preventDefault();
            }
        });
        // Blur on pressing escape needs to be bound to keyup and not keydown,
        // otherwise the keyup will not trigger
        this.$searchField.on('keyup.repository-search', function(e) {
            var dialogVisible = repositorySearch.dialog.is(':visible');
            if (!dialogVisible && e.which === AJS.keyCode.ESCAPE) {
                $(this).blur();
            }
        });
        this.$searchField.on('focus', function() {
            events.trigger('bitbucket.internal.ui.nav.search.focused');
        });
    };

    RepositorySearch.prototype._parseSearchTerm = function(searchTerm) {
        searchTerm = $.trim(searchTerm);
        if (searchTerm === '' || searchTerm === '/') {
            return null; // don't search
        }

        var searchRequest = {};
        var indexOfSeparator = searchTerm.lastIndexOf('/');
        if (indexOfSeparator === -1) {
            searchRequest.projectName = null;
            searchRequest.repositoryName = searchTerm;
        } else {
            searchRequest.projectName = $.trim(searchTerm.substr(0, indexOfSeparator));
            searchRequest.repositoryName = $.trim(searchTerm.substr(indexOfSeparator + 1));
        }
        return searchRequest;
    };

    // @override
    RepositorySearch.prototype._showSpinner = function(scrollable) {
        // Ensure the one in the scrollable is hidden
        scrollable.$scrollElement.find('.spinner').hide();

        this.$searchField.closest('.aui-quicksearch').addClass('loading');
        this.$searchField.next('.spinner').spin();
    };

    // @override
    RepositorySearch.prototype._hideSpinner = function() {
        this.$searchField.closest('.aui-quicksearch').removeClass('loading');
        this.$searchField.next('.spinner').spinStop();
    };

    return RepositorySearch;
});
