define(
    'bitbucket/internal/bbui/bb-panel',
    ['exports', 'module', 'jquery', 'aui', 'skate'],
    function (exports, module, _jquery, _aui, _skate) {
        'use strict';

        function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

        var _$ = _interopRequireDefault(_jquery);

        var _AJS = _interopRequireDefault(_aui);

        var _skate2 = _interopRequireDefault(_skate);

        /**
         * @param {BBPanel} el
         * @returns {Element|null} - The anchor element for this panel if it exists, or {@like null}
         */
        function getAnchor(el) {
            return document.getElementById(el.getAttribute('anchor-to'));
        }

        /**
         * @param {BBPanel} el - A BBPanel
         * @param {Function<Element>} callback - The callback to call with the anchor.
         */
        function doIfAnchor(el, callback) {
            var anchor = getAnchor(el);

            if (anchor) {
                callback(anchor);
            }
        }

        /**
         * Asks layer manager to display the panel
         *
         * @param {BBPanel} el - A BBPanel
         */
        function showPanel(el) {
            if (el._isProcessingShow) {
                return;
            }

            doIfAnchor(el, function (anchor) {
                // TODO consider the window resizing...
                var $anchor = (0, _$['default'])(anchor);
                var offset = $anchor.offset();
                var xOffset = offset.left;
                var yOffset = offset.top + $anchor.outerHeight();
                el.style.transform = 'translate3d(' + xOffset + 'px, ' + yOffset + 'px, 0)';
            });

            // This is to avoid being registered twice by the layer manager.
            // While it is processing the show it will set aira-hidden causing an update to bb-panel.
            el._isProcessingShow = true;
            _AJS['default'].layer(el).show();
            el._isProcessingShow = false;
        }

        /**
         * Requests layer manager hides the panel
         *
         * @param {BBPanel} el
         */
        function hidePanel(el) {
            _AJS['default'].layer(el).hide();
        }

        /**
         * Ensures the panels real state of showing or not matches its attributes
         * If the state doesn't match the panel is shown or hidden to get them in sync.
         *
         * @param {BBPanel} el
         */
        function reflectOpenness(el) {
            var isInitalizing = !el.hasAttribute('aria-hidden');
            var shouldBeOpen = el.hasAttribute('open');
            if (isInitalizing || el.open !== shouldBeOpen) {
                if (shouldBeOpen) {
                    showPanel(el);
                } else {
                    hidePanel(el);
                }
            }
        }

        /**
         * Sets a boolean attribute on el
         * If the value is {@link true} the attribute will exist but have no value,
         * If the value is {@link false} the attribute is removed
         *
         * @param {Element} el
         * @param {String} attr
         * @param {Boolean} newValue
         */
        function setBooleanAttribute(el, attr, newValue) {
            if (newValue) {
                el.setAttribute(attr, '');
            } else {
                el.removeAttribute(attr);
            }
        }

        var BBPanel = (0, _skate2['default'])('bb-panel', {
            prototype: Object.defineProperties({}, {
                open: {
                    /**
                     * Returns whether the inline dialog is open.
                     */

                    get: function get() {
                        return _AJS['default'].layer(this).isVisible();
                    },

                    /**
                     * Opens or closes the panel, returning whether the panel is open or closed as a result
                     * (since event handlers can prevent either action).
                     *
                     * You should check the value of open after setting this
                     * value since the before show/hide events may have prevented it.
                     */
                    set: function set(value) {
                        // TODO AUI-3726 Revisit double calls to canceled event handlers.
                        // Explicitly calling reflectOpenness(…) in this setter means
                        // that in native we'll get two sync calls to reflectOpenness(…)
                        // and in polyfill one sync (here) and one async (attr change
                        // handler). The latter of the two calls, for both cases, will
                        // usually be a noop (except when show/hide events are cancelled).
                        setBooleanAttribute(this, 'open', value);
                        reflectOpenness(this);
                    },
                    configurable: true,
                    enumerable: true
                }
            }),

            created: function created(element) {
                element._isProcessingShow = false;
            },

            attributes: {
                'aria-hidden': function ariaHidden(element, change) {
                    // Whenever layer manager hides us, we need to sync the open attribute.
                    setBooleanAttribute(element, 'open', change.newValue === 'false');
                },

                open: function open(element, change) {
                    // skate runs the created callback for attributes before the
                    // element is attached to the DOM, so guard against that.
                    if (document.body.contains(element)) {
                        reflectOpenness(element);
                    }
                }
            },

            attached: function attached(element) {
                reflectOpenness(element);
            },

            template: function template(element) {
                (0, _$['default'])(element).addClass('aui-layer');
            }
        });

        module.exports = BBPanel;
    }
);