import Widget from 'bitbucket/internal/widget';
import FilterBar from 'bitbucket/internal/filter-bar';
import PullRequestTable from 'bitbucket/internal/pull-request-table';
import PullRequestListDataProvider from 'bitbucket/internal/data-provider/pull-request-list';
import UserDataProvider from 'bitbucket/internal/user-data-provider';
import RefDataProvider from 'bitbucket/internal/ref-data-provider';
import models from 'bitbucket/internal/models';
import AJS from 'aui';
import $ from 'jquery';
import _ from 'lodash';
import openInSameTab from './dom-event';

class PullRequestList extends Widget {
    /**
     * @param {jQuery|HTMLElement} el
     * @param {Object} options
     * @param {Repository?} options.repository
     * @param {User?} options.currentUser
     * @param {Array<PullRequest>?} options.initialData - if you'd like to pass in a prefetched page of PRs, this is it.
     * @param {Object?} options.filter
     * @param {boolean} options.filter.reviewer_self - true to filter to just ones where the current user is a reviewer.
     * @param {models.PullRequestState?} options.filter.state
     * @param {Ref?} options.selectedTargetBranch
     * @param {Participant?} options.selectedAuthor
     */
    constructor(el, options) {
        super(options);

        options.filter = options.filter || {};
        options.filter.author_id = options.selectedAuthor && options.selectedAuthor.name;
        options.filter.target_ref_id = options.selectedTargetBranch && options.selectedTargetBranch.id;

        // we probably should make this reusable, but since this is the only place that needs it, I've punted on doing that for now.
        // The next person who want sto do preloading should extract this into something shareable (e.g., perhaps some kind of mixin instead of inheriting from the user-specific provider)
        class PreloadingUserDataProvider extends UserDataProvider {
            constructor(options, ...args) {
                super(options, ...args);

                this._preloaded = false;
                this._preloadItems = options.preload || [];
                this._equalityCheck = options.equals || function(a, b) {
                    return a.id === b.id;
                };
            }

            reset() {
                this._preloaded = false;
                return super.reset();
            }

            _fetchNext(lastResponseData) {
                if (!this._preloaded) {
                    if (this.filter.term) {
                        this._preloaded = true;
                    } else {
                        var promise = $.Deferred().resolve(this._preloadItems);
                        promise.abort = $.noop;
                        return promise;
                    }
                }

                return super._fetchNext(lastResponseData === this._preloadItems ? null : lastResponseData);
            }

            _transform(data) {
                if (!this._preloaded) {
                    this._preloaded = true;
                    return this._preloadItems;
                }

                let equals = this._equalityCheck;
                let preloadItems = this._preloadItems;
                var out = super._transform(data);
                if (this.filter.term) {
                    return out;
                } else {
                    // if we're not filtering, exclude the preloaded items from the output
                    return out.filter(function(item) {
                        return !preloadItems.some(function(preloadItem) {
                            return equals(preloadItem, item);
                        });
                    });
                }
            }
        }

        let authorProvider = new PreloadingUserDataProvider({
            preload: [options.currentUser],
            equals: function(a, b) { return a.name === b.name; },
            filter: {
                permissions: [{
                    name: 'REPO_READ',
                    repository_id: options.repository.id
                }]
            }
        });

        let branchProvider = new RefDataProvider({
            filter: {
                repository: options.repository,
                type: 'branch'
            }
        });

        let prProvider = new PullRequestListDataProvider({
            repository: options.repository.id,
            filter: {
                state: options.filter.state || models.PullRequestState.OPEN,
                author_id: options.filter.author_id || null,
                target_ref_id: options.filter.target_ref_id || null,
                reviewer_id: options.filter.reviewer_self ? options.currentUser.name : null
            }
        }, options.initialData);

        let pullRequestListFilters = [
            {
                id: 'pr-state-filter',
                type: FilterBar.Types.SELECT,
                text: AJS.I18n.getText('bitbucket.component.pull.request.list.state'),
                value: options.filter.state,
                menu: {
                    items:  [
                        {id: models.PullRequestState.OPEN, text: AJS.I18n.getText('bitbucket.component.pull.request.list.state.open')},
                        {id: models.PullRequestState.MERGED, text: AJS.I18n.getText('bitbucket.component.pull.request.list.state.merged')},
                        {id: models.PullRequestState.DECLINED, text: AJS.I18n.getText('bitbucket.component.pull.request.list.state.declined')}
                    ]
                }
            }, {
                type: FilterBar.Types.SELECT,
                text: AJS.I18n.getText('bitbucket.component.pull.request.list.author'),
                id: 'pr-author-filter',
                value: options.filter.author_id || null,
                searchPlaceholder: AJS.I18n.getText('bitbucket.component.pull.request.list.search.author'),
                menu: {
                    dataProvider: authorProvider,
                    id: function(user) { return user.name; },
                    initSelection: function($el, callback) {
                        var username = $el.val();
                        if (options.filter.author_id === username) {
                            return callback(options.selectedAuthor);
                        }
                        throw new Error(`Unexpected value '${username}' when initializing the author filter.`);
                    },
                    formatSelection: function(author, container) {
                        return bitbucket.component.pullRequestList.authorSelection({author});
                    },
                    formatResult:  function(author, container) {
                        return bitbucket.component.pullRequestList.authorResult({author});
                    },
                    formatNoMatches: function() {
                        return AJS.I18n.getText('bitbucket.component.pull.request.list.search.author.nomatches');
                    },
                    placeholder: AJS.I18n.getText('bitbucket.component.pull.request.list.author'),
                    dropdownCssClass: 'pr-author-dropdown'
                }
            }, {
                type: FilterBar.Types.SELECT,
                text:  AJS.I18n.getText('bitbucket.component.pull.request.list.branch.target'),
                id: 'pr-target-branch-filter',
                value: options.filter.target_ref_id || null,
                searchPlaceholder: AJS.I18n.getText('bitbucket.component.pull.request.list.search.branch'),
                menu: {
                    dataProvider: branchProvider,
                    initSelection: function($el, callback) {
                        var branch = $el.val();
                        if (options.filter.target_ref_id === branch) {
                            return callback(options.selectedTargetBranch);
                        }
                        throw new Error(`Unexpected value '${branch}' when initializing the target branch filter.`);
                    },
                    formatSelection: function(branch, container) {
                        return bitbucket.component.pullRequestList.targetBranchSelection({branch});
                    },
                    formatResult:  function(branch, container) {
                        return bitbucket.component.pullRequestList.targetBranchResult({branch});
                    },
                    formatNoMatches: function() {
                        return AJS.I18n.getText('bitbucket.component.pull.request.list.search.branch.nomatches');
                    },
                    placeholder: AJS.I18n.getText('bitbucket.component.pull.request.list.branch.target'),
                    dropdownCssClass: 'pr-target-branch-dropdown'
                }
            }
        ];

        if (options.currentUser) { // anonymous users shouldn't see the "I'm reviewing"
            pullRequestListFilters.push({
                type: FilterBar.Types.TOGGLE,
                text:  AJS.I18n.getText('bitbucket.component.pull.request.list.reviewer.self'),
                id: 'pr-reviewer-self-filter',
                value: options.filter.reviewer_self
            });
        }

        let filterBar = new FilterBar(el, {
            id: 'pull-requests-filter-bar',
            filters: pullRequestListFilters
        });

        this.pullRequestTable = new PullRequestTable(el, {
            id: 'pull-requests-table',
            dataProvider: prProvider,
            isFiltered: () => {
                var filter = prProvider.filter;
                return (filter.state !== 'OPEN' || filter.author_id || filter.target_ref_id || filter.reviewer_id);
            },
            repository: options.repository,
            noneMatchingMessageHtml: bitbucket.component.pullRequestList.noneMatchingMsg({ repository: options.repository })
        });

        this.pullRequestTable.init();

        let resetFilters = () => {
            filterBar.state = _.transform(pullRequestListFilters, (state, filter) => {
                state[filter.id] = (filter.menu && filter.menu.items) ? filter.menu.items[0].id : false;
            }, {});
        };

        $(el).on('click', '#reset-filters', (e) => {
            if (openInSameTab(e)) {
                e.preventDefault();
                resetFilters();
            }
        });

        filterBar.on('change', () => {
            let filterState = filterBar.state;
            let cleanState = {
                state: filterState['pr-state-filter'],
                author_id: filterState['pr-author-filter'] || null,
                target_ref_id: filterState['pr-target-branch-filter'] || null,
                reviewer_self: filterState['pr-reviewer-self-filter']
            };

            prProvider.setFilter('state', cleanState.state);
            prProvider.setFilter('author_id', cleanState.author_id);
            prProvider.setFilter('target_ref_id', cleanState.target_ref_id);
            prProvider.setFilter('reviewer_id', cleanState.reviewer_self ? options.currentUser.name : null);
            prProvider.reset();

            this.trigger('state-change', cleanState);
        });

        // if there isn't a focused row set focus to the inital row in the table.
        let focusInitialRowIfNeeded = () => {
            if (!this.pullRequestTable._getFocusedItem().length) {
                this.pullRequestTable.focusInitialRow();
            }
        };
        // when a page is rendered, check if we need to focus the initial row.
        this.pullRequestTable.on('page-rendered', focusInitialRowIfNeeded);
        focusInitialRowIfNeeded();

    }

    get _keyboardActions () {
        return {
            moveNext: this.pullRequestTable.moveNext,
            movePrevious: this.pullRequestTable.movePrevious,
            openItem: this.pullRequestTable.openItem
        };
    }

    getKeyboardAction (action) {
        let kbdAction = this._keyboardActions[action];
        if (typeof kbdAction === 'function') {
            return kbdAction;
        }
        throw new Error(`No keyboard action has been defined for '${action}'.`);
    }
}

export default PullRequestList;
