import PagedTable from 'bitbucket/internal/paged-table';
import Widget from 'bitbucket/internal/widget';
import $ from 'jquery';
import _ from 'lodash';

QUnit.module('bitbucket-paged-table', {
    setup: function () {
        $('#qunit-fixture').append('<table id="test-table"></table>');
    }
});

$.fn.spin = sinon.spy();
$.fn.spinStop = sinon.spy();

function createPage (start, size, isLastPage, nextPageStart) {
    return $.extend({
        start: start,
        size: size,
        isLastPage: isLastPage
    }, nextPageStart ? {
        nextPageStart: nextPageStart
    } : {});
}

function createTable (page) {
    return $('<table></table>')
        .attr($.extend({
                  'data-start': String(page.start),
                  'data-size': String(page.size),
                  'data-lastpage': String(page.isLastPage)
              }, page.nextPageStart ? {
                  'data-nextpagestart': String(page.nextPageStart)
              } : {}));
}

class DummyDataProvider extends Widget {
    constructor () {
        this.isFirstPage = true;
        sinon.spy(this, 'abort');
        sinon.spy(this, 'on');
        sinon.spy(this, 'reset');

        this.filter = {};
    }

    setFilter(k, v) {
        this.filter[k] = v;
    }

    abort() {

    }

    reset() {
        this.trigger('reset');
        this._page = -1;
    }

    fetchNext (shouldSucceed = true, timeout = 50) {
        this.data = this.data || [];
        this.trigger('data-requested');
        let promise = $.Deferred();
        setTimeout(() => {
            if (shouldSucceed) {
                promise.resolve(this.data);
            } else {
                promise.reject(this.data);
            }
        }, timeout);
        promise.done(() => this.trigger('data-loaded'));
        promise.fail(() => this.trigger('data-request-failed'));
        return promise;
    }
}

QUnit.test('Spinner appears during a request', function (assert) {
    let pagedTable = new PagedTable({
        tableEl: '#test-table',
        dataProvider: new DummyDataProvider()
    });
    let done = assert.async();
    pagedTable.provider.fetchNext(true).done(() => {
        // defer so event can fire
        _.defer(() => {
            assert.ok(pagedTable.$spinner.hasClass('hidden'), 'spinner is hidden after request finishes.');
            done();
        });
    });
    assert.ok(!pagedTable.$spinner.hasClass('hidden'), 'spinner is visible when a request is being made');

});

QUnit.test('Spinner disappears if request fails', function (assert) {
    let pagedTable = new PagedTable({
        tableEl: '#test-table',
        dataProvider: new DummyDataProvider()
    });
    let done = assert.async();
    pagedTable.provider.fetchNext(false).fail(() => {
        _.defer(() => {
            assert.ok(pagedTable.$spinner.hasClass('hidden'), 'spinner is hidden after request fails.');
            done();
        });
    });
    assert.ok(!pagedTable.$spinner.hasClass('hidden'), 'spinner is visible when a request is being made');
});

QUnit.test('changing the filter updates the dataProvider and resets it', function (assert) {
    let pagedTable = new PagedTable({
        tableEl: '#test-table',
        dataProvider: new DummyDataProvider(),
        filterable: true,
        filterEl: '<input id="filter-input" class="paged-table-filter-input" data-for="test-table" value="foo" />',
        filterDebounce: 0
    });
    sinon.spy(pagedTable, 'clear');

    let done = assert.async();
    let keyupEvent = new $.Event('keyup');
    pagedTable.$filter.trigger(keyupEvent);

    _.defer(() => {
        assert.equal(pagedTable.provider.filter.term, 'foo', 'data provider filter is updated to the UI value');
        assert.ok(pagedTable.clear.called, 'Filter change causes paged table to clear.');
        done();
    });

});

QUnit.test('No results message is shown when there are no results', function (assert) {
    let pagedTable = new PagedTable({
        tableEl: '#test-table',
        dataProvider: new DummyDataProvider()
    });
    sinon.spy(pagedTable, 'handleLastPage');
    pagedTable.handleNewRows = sinon.spy();
    pagedTable.handleNoData = sinon.spy();

    let done1 = assert.async();
    let done2 = assert.async();
    pagedTable.provider.reachedEnd = true;

    // Check the last page handler.
    function firstRequest () {
        pagedTable.provider.data = [1, 2, 3];
        return pagedTable.load().done(() => {
            assert.ok(pagedTable.handleLastPage.called, 'last page handler should be called');
            done1();
        });
    }

    // Check no data handler not called on last page.
    function secondRequest () {
        pagedTable.provider.data = [];
        return pagedTable.load().done(() => {
            assert.ok(pagedTable.handleNoData.notCalled, 'no data handler should not be called on the last page');
            done2();
        });
    }

    // check the no data handler
    function thirdRequest () {
        pagedTable.reset(); // resets _page to -1
        pagedTable.provider.data = [];
        return pagedTable.load().done(() => {
            assert.ok(pagedTable.handleNoData.called, 'when on the first page and there is no data, handleNoData should be called');
        });
    }

    firstRequest().then(() => {
        return secondRequest();
    }).then(() => {
        return thirdRequest();
    });

});

QUnit.test('handleNewRows is called with dataProvider response', function (assert) {
    let pagedTable = new PagedTable({
        tableEl: '#test-table',
        dataProvider: new DummyDataProvider()
    });
    pagedTable.handleNewRows = sinon.spy();

    let done = assert.async();
    pagedTable.provider.data = [1, 2, 3];
    pagedTable.load().done(() => {
        assert.ok(pagedTable.handleNewRows.called, 'handleNewRows is called when fetching data');
        assert.ok(pagedTable.handleNewRows.calledWith([1, 2, 3]), 'handleNewRows is called with the data from the data provider fetch');
        done();
    });
});
