(function (factory) {
    if (typeof define === 'function' && define.amd) {
        define(['exports', 'module', 'bitbucket/internal/widget', 'bitbucket/internal/json-validation', 'bitbucket/internal/javascript-errors', 'jquery'], factory);
    } else if (typeof define === 'function' && !define.amd) {
        define('bitbucket/internal/spi/data-provider', ['exports', 'module', 'bitbucket/internal/widget', 'bitbucket/internal/json-validation', 'bitbucket/internal/javascript-errors', 'jquery'], factory);
    } else if (typeof exports !== 'undefined') {
        factory(exports, module);
    }
})(function (exports, module) {

"use strict";

var _createClass = (function () { function defineProperties(target, props) { for (var key in props) { var prop = props[key]; prop.configurable = true; if (prop.value) prop.writable = true; } Object.defineProperties(target, props); } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; })();

var _get = function get(object, property, receiver) { var desc = Object.getOwnPropertyDescriptor(object, property); if (desc === undefined) { var parent = Object.getPrototypeOf(object); if (parent === null) { return undefined; } else { return get(parent, property, receiver); } } else if ("value" in desc && desc.writable) { return desc.value; } else { var getter = desc.get; if (getter === undefined) { return undefined; } return getter.call(receiver); } };

var _inherits = function (subClass, superClass) { if (typeof superClass !== "function" && superClass !== null) { throw new TypeError("Super expression must either be null or a function, not " + typeof superClass); } subClass.prototype = Object.create(superClass && superClass.prototype, { constructor: { value: subClass, enumerable: false, writable: true, configurable: true } }); if (superClass) subClass.__proto__ = superClass; };

var _classCallCheck = function (instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } };

var Widget = require("bitbucket/internal/widget");
var validator = require("bitbucket/internal/json-validation");
var errors = require("bitbucket/internal/javascript-errors");
var $ = require("jquery");

var index__NotImplementedError = errors.NotImplementedError;

/**
 * The DataProvider is a relatively abstract class that provides some basic data fetching and management.
 * Each product should implement its own version of this to then provide to consumers (components).
 *
 * Data provided to a DataProvider is expect to at the least have an array of values along with some metadata.
 * The metadata will differ per product and this will be dealt with in the product specific implementations.
 *
 */

var DataProvider = (function (_Widget) {

    /**
     * @param {Object} options - The options for the Data Provider
     * @param {Object?} options.jsonDescriptor - If provided, transformed data will be validated against this shape.
     *                                           See the rules in declarative-validation.js and reference the unit tests.
                                                 If not provided, _validate must be implemented in a subclass.
     * @param {Object?} initialData - The initial data for this provider
     */

    function DataProvider() {
        var options = arguments[0] === undefined ? {} : arguments[0];
        var initialData = arguments[1] === undefined ? null : arguments[1];

        _classCallCheck(this, DataProvider);

        _get(Object.getPrototypeOf(DataProvider.prototype), "constructor", this).call(this, options);
        this.initialData = initialData;

        if (this.options.jsonDescriptor) {
            this._validate = validator(this.options.jsonDescriptor);
        }
    }

    _inherits(DataProvider, _Widget);

    _createClass(DataProvider, {
        isFetching: {

            /**
             * If the data provider is currently fetching data with an XHR request, this will return true.
             *
             * Checks for the presence of the DataProvider#_requestPromise
             *
             * @returns {boolean}
             */

            get: function () {
                return !!this._requestPromise;
            }
        },
        reset: {

            /**
             * Reset the data for this provider.
             */

            value: function reset() {
                this.abort();
                this.currentData = null;
                delete this.initialData;
                this.trigger("reset");
            }
        },
        abort: {

            /**
             * Cancel any in-flight requests and nullify the request promise
             */

            value: function abort() {
                if (this._requestPromise) {
                    this._requestPromise.abort();
                    this._requestPromise = null;
                    this.trigger("abort");
                }
            }
        },
        fetch: {

            /**
             * Perform the AJAX request for this DataProvider.
             * If called and there is currentData use the locally cached version.
             *
             * This means that {DataProvider#currentData} needs to be invalidated if
             * you want to force the fetching of data from the server on subsequent requests.
             *
             * @param {string?} url - the url to fetch data from. defaults to the base url
             *
             * @returns {Promise}
             */

            value: function fetch() {
                var _this = this;

                var url = arguments[0] === undefined ? this.url : arguments[0];

                // If a fetch is attempted before the previous fetch has returned,
                // then abort the previous fetch and continue with the new request.
                this.abort();

                // If there is initial data, upgrade the currentData to initialData
                if (this.initialData) {
                    this.currentData = this.transform(this.initialData);
                    // initialData is single use, delete it after it has been consumed
                    delete this.initialData;
                }

                // If there is a set of currentData then return it.
                if (this.currentData) {
                    return $.Deferred().resolve(this.currentData);
                }

                this.trigger("data-requested");

                this._requestPromise = this._fetch(url);

                // The promise from _fetch must be abortable.
                if (!this._requestPromise.abort) {
                    throw new Error("no abort method on DataProvider#_requestPromise.");
                }

                return this._requestPromise.then(this.transform, function () {
                    for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
                        args[_key] = arguments[_key];
                    }

                    return $.Deferred().reject(_this.errorTransform.apply(_this, args));
                }).done(function (transformedData) {
                    _this.trigger("data-loaded", transformedData);
                    _this.currentData = transformedData;
                }).fail(function (error) {
                    _this.trigger("data-request-failed", error);
                }).always(function () {
                    _this._requestPromise = null;
                });
            }
        },
        _fetch: {

            /**
             * Perform a request to fetch data.
             *
             * @returns {Promise} - a jqxhr promise. This promise should be abortable.
             * @abstract
             * @protected
             */

            value: function _fetch() {
                throw new index__NotImplementedError();
            }
        },
        transform: {

            /**
             * Transform the results from a REST request to the format required
             * by a consumer of a component data provider.
             */

            value: function transform() {
                for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
                    args[_key] = arguments[_key];
                }

                var out = this._transform.apply(this, args);
                this._validate(out);
                return out;
            }
        },
        _transform: {

            /**
             * @returns {Object}
             * @protected
             */

            value: function _transform() {
                throw new index__NotImplementedError();
            }
        },
        _validate: {

            /**
             * Throw an error if the transformed data doesn't have the required shape.
             * If a jsonDescriptor is provided, it will be used for validation.
             * @protected
             */

            value: function _validate(data) {
                throw new index__NotImplementedError();
            }
        },
        errorTransform: {

            /**
             * Transform the errors from a REST request to the format required
             * by a consumer of a component data provider.
             *
             * @returns {Object|string}
             */

            value: function errorTransform() {
                for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
                    args[_key] = arguments[_key];
                }

                return this._errorTransform.apply(this, args);
            }
        },
        _errorTransform: {

            /**
             * @returns {Object|string}
             * @protected
             */

            value: function _errorTransform() {
                throw new index__NotImplementedError();
            }
        }
    });

    return DataProvider;
})(Widget);

var index = DataProvider;

module.exports = index;

});
