(function (factory) {
    if (typeof define === 'function' && define.amd) {
        define(['exports', 'module', 'lodash'], factory);
    } else if (typeof define === 'function' && !define.amd) {
        define('bitbucket/internal/json-validation', ['exports', 'module', 'lodash'], factory);
    } else if (typeof exports !== 'undefined') {
        factory(exports, module);
    }
})(function (exports, module) {

"use strict";

var _ = require("lodash");

function validator(type) {
    if (!type) {
        // false|null|undefined|0
        return _.identity;
    }

    if (typeof type === "string" || type instanceof String) {
        type += "";
        var nullableIndex = type.indexOf("?");
        var nullable = false;
        if (nullableIndex !== -1) {
            if (nullableIndex !== 0 && nullableIndex !== type.length - 1) {
                throw new Error("Type unexpectedly contains a \"?\" in the middle: " + type);
            }
            nullable = true;
            type = type.split("?")[nullableIndex ? 0 : 1];
        }
        return function (value) {
            if (value == null) {
                if (nullable) {
                    return;
                }
                throw new Error("Value was null, but expected non-nullable type " + type);
            }
            if (typeof value !== type && !(typeof value === "function" && type === "object")) {
                throw new Error("The typeof " + value + " is " + typeof value + " but expected it to be " + type + (nullable ? " or value to be null." : "."));
            }
        };
    }

    if (_.isArray(type) || type === Array) {
        var elementCheck = type !== Array && type[0] ? validator(type[0]) : _.identity;
        return function (value) {
            if (!_.isArray(value)) {
                throw new Error("Array type expected, but null or non-Array value provided.");
            }
            value.forEach(elementCheck);
        };
    }

    if (type === Object || type && typeof type === "object") {
        var validations = type === Object ? [] : Object.keys(type).map(function (typeKey) {
            return [typeKey, validator(type[typeKey])];
        });

        return function (value) {
            if ((!value || typeof value !== "object") && typeof value !== "function") {
                throw new Error("Object expected, but null or non-Object value provided.");
            }
            validations.forEach(function (validation) {
                var typeKey = validation[0];
                var func = validation[1];
                func(value[typeKey]);
            });
        };
    }

    if (typeof type === "function") {
        // must come after Array and Object - those constructors are functions!
        return type;
    }

    throw new Error("Invalid expected type " + type + ". Should be falsy, String, Array, Object, or function.");
}

/**
 * If you are expecting the value to be one of N known strings, use this helper.
 */
validator.asEnum = function (name, enumeration) {
    return function validateEnum(val) {
        var hasValue = Object.keys(enumeration).some(function (key) {
            return enumeration[key] === val;
        });
        if (hasValue) {
            return;
        }
        throw new Error("Invalid " + name + ". Expected one of: " + Object.keys(enumeration).join(", "));
    };
};

/**
 * You can do nullable primitives with 'string?', but for nullable objects, use this helper.
 */
validator.nullable = function (obj) {
    var objValidator = validator(obj);
    return function (v) {
        if (v == null) {
            return;
        }
        objValidator(v);
    };
};

/**
 * If you want to defer the creation of the validation function until you actually request a validation, use this helper.
 */
validator.recurse = function (getModel) {
    var validate;
    return function (value) {
        if (!validate) {
            validate = validator(getModel());
        }
        validate(value);
    };
};

/**
 * If there is only one valid value for a field, use this helper.
 */
validator.strictEqual = function (value) {
    return function (v) {
        if (v !== value) {
            throw new Error("Expected " + value + " but was " + v);
        }
    };
};

module.exports = validator;

});
