(function (factory) {
    if (typeof define === 'function' && define.amd) {
        define(['exports', 'module', 'bitbucket/internal/javascript-errors'], factory);
    } else if (typeof define === 'function' && !define.amd) {
        define('bitbucket/internal/spi/web-fragments', ['exports', 'module', 'bitbucket/internal/javascript-errors'], factory);
    } else if (typeof exports !== 'undefined') {
        factory(exports, module);
    }
})(function (exports, module) {

"use strict";

var _createClass = (function () { function defineProperties(target, props) { for (var key in props) { var prop = props[key]; prop.configurable = true; if (prop.value) prop.writable = true; } Object.defineProperties(target, props); } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; })();

var _classCallCheck = function (instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } };

var errors = require("bitbucket/internal/javascript-errors");

var NotImplementedError = errors.NotImplementedError;

/**
 * @enum WebFragmentType
 */
var FragmentType = {
    ITEM: "ITEM",
    SECTION: "SECTION",
    PANEL: "PANEL"
};

/**
 * @class WebItem
 * @description Generally for displaying a link, but useful for any case where the plugin should not be allowed to determine the generated markup. E.g. a toolbar button or menu item, etc.
 * @property {string} type - must be ITEM
 * @property {number} [weight=1000] - a number used to order the item. Lower weights appear earlier or more prominently in the UI than higher ones.
 * @property {string} location - the location this item is being injected
 * @property {string} url - a URL to link to
 * @property {string} key - a unique identifier for the item (in Bitbucket Server this is based on completeModuleKey)
 * @property {string} text - text for the item
 * @property {Map<string,string>} params - a map of parameter name to value. Useful for any custom configuration required for items in a particular location.
 * @property {string?} id - an ID to be applied to the item. Useful for JS-triggered items in Bitbucket Server.
 * @property {string?} cssClass - a CSS class to be applied to the item. Useful for JS-triggered items in Bitbucket Server.
 * @property {string?} tooltip - a tooltip to display on the item
 * @property {string?} iconUrl - a URL for an icon to display with the item
 * @property {number?} iconWidth - a px value for the width at which to display the icon
 * @property {number?} iconHeight - a px value for the height at which to display the icon
 */

/**
 * @class WebSection
 * @description A grouping of other web fragments. The convention for usage of a Web Section is that its key is used as the location for another fragment request.
 * @property {string} type - must be SECTION
 * @property {number} [weight=1000] - a number used to order the section. Lower weights appear earlier or more prominently in the UI than higher ones.
 * @property {string} location - the location this section is being injected
 * @property {string} key - a unique identifier for the section (in Bitbucket Server this is based on completeModuleKey)
 * @property {string?} text - text for the section
 * @property {Map<string,string>} params - a map of parameter name to value. Useful for any custom configuration required for sections in a particular location.
 */

/**
 * @class WebPanel
 * @description Arbitrary markup. Via Connect, this will general be an iframe containing plugin content.
 * @property {string} type - must be PANEL
 * @property {number} [weight=1000] - a number used to order the panel. Lower weights appear earlier or more prominently in the UI than higher ones.
 * @property {string} location - the location this panel is being injected
 * @property {string} key - a unique identifier for the panel (in Bitbucket Server this is based on completeModuleKey)
 * @property {string} html - the html to be injected
 * @property {Map<string,string> params - a map of parameter name to value. Useful for any custom configuration required for panels in a particular location.
 */

/**
 * An SPI that provides web fragments
 */

var WebFragments = (function () {
    function WebFragments() {
        _classCallCheck(this, WebFragments);
    }

    _createClass(WebFragments, {
        getWebItems: {
            /**
             * Return web items for a location and context
             *
             * @param {string} location - the lookup string to get plugin items for
             * @param {Object} context - an object that can be provided to plugins to help them generate the right content
             * @returns {Array<WebItem>} sorted by weight
             */

            value: function getWebItems(location, context) {
                throw new NotImplementedError();
            }
        },
        getWebSections: {

            /**
             * Return web sections for a location and context
             *
             * @param {string} location - the lookup string to get plugin sections for
             * @param {Object} context - an object that can be provided to plugins to help them generate the right content
             * @returns {Array<WebSection>} sorted by weight
             */

            value: function getWebSections(location, context) {
                throw new NotImplementedError();
            }
        },
        getWebPanels: {

            /**
             * Return web panels for a location and context
             *
             * @param {string} location - the lookup string to get plugin panels for
             * @param {Object} context - an object that can be provided to plugins to help them generate the right content
             * @returns {Array<WebPanel>} sorted by weight
             */

            value: function getWebPanels(location, context) {
                throw new NotImplementedError();
            }
        },
        getWebFragments: {

            /**
             * Return web items, sections, and/or panels for a location
             *
             * @param {string} location - the lookup string to get plugin fragments for
             * @param {Array<WebFragmentType>} types - an array of types you want to retrieve - 'ITEM', 'SECTION', or 'PANEL'
             * @param {Object} context - an object that can be provided to plugins to help them generate the right content
             * @returns {Array<WebItem|WebSection|WebPanel>} sorted by weight
             */

            value: function getWebFragments(location, types, context) {
                throw new NotImplementedError();
            }
        }
    }, {
        getExampleItem: {

            /**
             * @param {string} location - the location that the item is for
             * @param {Object} context - the context for the location
             * @returns {WebItem}
             */

            value: function getExampleItem(location, context) {
                return {
                    type: "ITEM",
                    weight: 1000,
                    location: location,
                    url: "#example-web-item-url",
                    key: "example-web-item",
                    text: "Client Web Item: " + location,
                    params: {},
                    id: null,
                    cssClass: "plugin-point",
                    tooltip: "Client Context Items: " + Object.keys(context).join(", "),
                    iconUrl: null,
                    iconWidth: 0,
                    iconHeight: 0
                };
            }
        },
        getExampleSection: {

            /**
             * @param {string} location - the location that the section is for
             * @param {Object} context - the context for the location
             * @returns {WebSection}
             */

            value: function getExampleSection(location, context) {
                return {
                    type: "SECTION",
                    weight: 1000,
                    location: location,
                    key: "example-web-section",
                    text: "Client Web Section: " + location,
                    params: {}
                };
            }
        },
        getExamplePanel: {

            /**
             * @param {string} location - the location that the panel is for
             * @param {Object} context - the context for the location
             * @returns {WebPanel}
             */

            value: function getExamplePanel(location, context) {
                return {
                    type: "PANEL",
                    weight: 1000,
                    location: location,
                    key: "example-web-panel",
                    html: "<div class=\"plugin-point web-panel\">" + "<strong>Client Web Panel</strong>: " + location + "<br />" + "<strong>Client Context Items</strong>: " + Object.keys(context).join(", ") + "</div>",
                    params: {}
                };
            }
        }
    });

    return WebFragments;
})();

var index = WebFragments;

module.exports = index;

});
