import Widget from 'bitbucket/internal/widget';
import AJS from 'aui';
import $ from 'jquery';
import _ from 'lodash';
import navigatorUtil from 'internal/util/navigator';

/**
 * The type of filter to show.
 * @enum {string} FilterType
 */
const FilterType = {
    SELECT: 'SELECT', // a dropdown menu - loaded synchronously or via AJAX
    TOGGLE: 'TOGGLE' // a toggle button - on or off
};

/**
 * @class FilterDescriptor
 * @property {FilterType} type - the type of this filter
 * @property {string} id - the HTML ID for the filter, as well as the key under which the filter's state will be returned. Must be unique on the page.
 * @property {string} text -
 */
function validateFilters (filters) {
    var filtersById = {};
    filters.forEach(function (filter) {
        ['type', 'id', 'text'].forEach(function (prop) {
            if (typeof filter[prop] !== 'string') {
                throw new Error('Every filter needs a ' + prop);
            }
        });
        if (filter.type === 'SELECT') {
            var menu = filter.menu;
            if (!menu) {
                throw new Error('select filters need a menu configuration.');
            }
            if (!menu.items) {
                if (!menu.dataProvider || !menu.placeholder) {
                    if (!menu.dataProvider && !menu.placeholder) {
                        throw new Error('select filters need a menu with either hardcoded items or dataProvider and placeholder.');
                    }
                    throw new Error("When items aren't provided, dataProvider and placeholder must be.");
                }
            } else if (menu.dataProvider) {
                throw new Error('items is mutually exclusive with dataProvider.');
            }
        }
        if (filtersById[filter.id]) {
            throw new Error('duplicated ID ' + filter.id);
        }
        filtersById[filter.id] = filter;
    });
}

function initToggle(filter, $filter) {
    $filter.click(function () {
        $filter.attr('aria-pressed', ($filter.attr('aria-pressed') && $filter.attr('aria-pressed') !== 'false') ? null : 'true');
        $filter.trigger('change');
    });
}

function initStaticSelect (filter, $filter) {
    let defaults = {
        minimumResultsForSearch: -1 // don't show the search box.
    };
    let overrides = {};
    $filter.auiSelect2(_.extend(defaults, filter.menu, overrides));
}

function initAsyncSelect (filter, $filter) {
    let menu = filter.menu;
    let dataProvider = menu.dataProvider;
    let $searchBoxSpinner = $(bitbucket.component.filterBar.loadMore());
    dataProvider.__test = 0;
    let defaults = {
        minimumInputLength: 0,
        dropdownAutoWidth: true,
        formatSearching: function () {
            return AJS.I18n.getText('bitbucket.component.filter.bar.searching');
        },
        formatNoMatches: function () {
            return AJS.I18n.getText('bitbucket.component.filter.bar.nomatches');
        }
    };
    let overrides = {
        allowClear: true,
        containerCssClass: `filter-bar-async ${menu.containerCssClass || ''}`.trim(),
        dropdownCssClass: `filter-bar-async filter-bar-dropdown-${filter.id} ${menu.dropdownCssClass || ''}`.trim(),
        query: function (query) {
            dataProvider.setFilter('term', query.term);

            if (query.page <= 1) { // first page for this search.
                $searchBoxSpinner.spin();
                dataProvider.reset();
            }

            let { more, promise } = dataProvider.fetchNext ? {
                more: () => !dataProvider.reachedEnd,
                promise: dataProvider.fetchNext()
            } : {
                more: () => false,
                promise: dataProvider.fetch()
            };

            let $spinner = $('.select2-more-results').html(bitbucket.component.filterBar.loadMore());
            $spinner.spin();

            promise.then(function (items) {
                $spinner.spinStop();
                $searchBoxSpinner.spinStop();
                query.callback({
                    context: query.context,
                    results: menu.transform ?
                        items.map(menu.transform) :
                        items,
                    more: more()
                });
            });
        },
        formatLoadMore: function () {
            return ' '; // blank out the default text because we're replacing it with the spinner
        },
        ajax: undefined
    };
    $filter.auiSelect2(_.extend(defaults, menu, overrides));
    $filter.on('select2-opening', function (e) {
        if ($filter.select2('val')) { // if there's a selection, force the clearing of it instead of opening.
            e.preventDefault();
            $filter.select2('val', '', true); // clear and trigger change
        }
    });
    if (filter.searchPlaceholder && !navigatorUtil.isIE()) { // IE11 & below will fail - see STASHDEV-10518
        $filter.one('select2-open', function () {
            // make sure it's filtered by specific dropdown so this only affects the select2 that $filter belongs to
            $(`.filter-bar-dropdown-${filter.id} .select2-search > input`)
                .attr('placeholder', filter.searchPlaceholder)
                .after($searchBoxSpinner);
        });
    }
}

class FilterBar extends Widget {
    /**
     * @param {jQuery|HTMLElement} el
     * @param {Object} options
     */
    constructor (el, options) {
        super(options);
        validateFilters(options.filters);

        var $filterBar = this._$filterBar = $(bitbucket.component.filterBar.main({
            id: options.id,
            filters: options.filters.map(function (filter) {
                if (!filter.menu || !filter.menu.item || !filter.menu.transform) {
                    return filter;
                }
                filter = _.deepClone(filter);
                filter.menu.items = filter.menu.items.map(filter.menu.transform);
                return filter;
            })
        }));

        options.filters.forEach(function (filter) {
            var $filter = $filterBar.find('#' + filter.id);
            switch (filter.type) {
                case FilterType.TOGGLE:
                    initToggle(filter, $filter);
                    break;
                case FilterType.SELECT:
                    if (filter.menu.items) { // static
                        initStaticSelect(filter, $filter);
                    } else { // data provider
                        initAsyncSelect(filter, $filter);
                    }
                    break;
            }
        });

        $(el).append($filterBar);
        $filterBar.on('change', () => {
            this.trigger('change');
        });
    }

    get state () {
        return _.transform(this.options.filters, (state, filter) => {
            var $filter = this._$filterBar.find('#' + filter.id);
            state[filter.id] = filter.type === FilterType.TOGGLE ?
                (!!$filter.attr('aria-pressed') && $filter.attr('aria-pressed') !== 'false') :
                $filter.val();
        }, {});
    }

    set state (filters) {
        _.map(filters, (state, filter) => {
            let $filter = this._$filterBar.find(`#${filter}`);
            if ($filter.is('button')) {
                $filter.attr('aria-pressed', state);
             } else {
                 $filter.select2('val', state);
            }
        });
        this.trigger('change');
    }

    static get Types () {
        return FilterType;
    }
}

export default FilterBar;
