define('bitbucket/internal/page/setupSettings', [
    'aui',
    'jquery',
    'lodash',
    'bitbucket/internal/util/client-storage',
    'bitbucket/internal/widget/setup-tracking',
    'exports',
    'aui/form-notification'
], function(
    AJS,
    $,
    _,
    clientStorage,
    setupTracking,
    exports
) {
    var hamletUrl;
    var lassoUrl;
    var browserSupportsCORS;
    var countries;

    var $settingsForm;
    var $licenseForm;
    var $document = $(document);

    exports.onReady = function(hamletUrlFromServer, lassoUrlFromServer) {
        /// Update globals
        hamletUrl = hamletUrlFromServer;
        lassoUrl = lassoUrlFromServer;
        $settingsForm = $("#settings"); // Top form with baseUrl, applicationTitle, serverId and a hidden license field
        $licenseForm = $('.license-form'); // Class of the bottom for where users can signin, signup or put their own key in. Class is used to hide these forms.

        browserSupportsCORS = _.once(function() {
            return "withCredentials" in new XMLHttpRequest();
        });

        var setupDataKey = clientStorage.buildKey([AJS.contextPath(), 'setup']);
        var setupData = clientStorage.getSessionItem(setupDataKey);

        if (setupData) {
            setupData.applicationTitle && $('#applicationTitle').val(setupData.applicationTitle);
            setupData.baseUrl && $('#baseUrl').val(setupData.baseUrl);
        }

        // Setup the country selector as an auiSelect2 box and populate it with countries
        updateCountries();

        $settingsForm.submit(function() {
            clientStorage.setSessionItem(setupDataKey, {
                applicationTitle: $('#applicationTitle').val(),
                baseUrl: $('#baseUrl').val()
            });
        });

        $document.on('keypress', '[data-aui-notification-error]', function(e){
            e.target.removeAttribute("data-aui-notification-error"); // Remove the error
        });

        $settingsForm.on('click', 'input[type="radio"]', function(e) {
            var $el = $(e.currentTarget);
            var elId = $el.attr("id");
            $licenseForm.hide();

            if(elId === "check-email-radio") {
                setupTracking.track('setup-settings-evaluate');
                $('#check-email').show();
                $('#check-email-email').focus();
            }

            if(elId === "has-key-radio") {
                setupTracking.track('setup-settings-has-key');
                $('#has-key').show();
                $('#licenseKey').focus();
            }
        });

        // When the submit button is clicked in the check-email form, lookup the email and redirect to the correct form
        $document.on('submit', '#check-email', function(e) {
            var $el = $(e.currentTarget);
            e.preventDefault();

            var data = extractFormData({'email': 'check-email-email'});

            // Strip whitespace so that call returns
            data.email = data.email && data.email.trim();

            // Make sure all required fields are filled in
            if (checkRequiredFields($el, data, bitbucket.internal.page.setup.checkEmail)) {
                return;
            }

            // Extra check that the email contains an "@"
            if (data.email.indexOf("@") < 0) {
                loadSoyTemplate($el, data, bitbucket.internal.page.setup.checkEmail, null, formatFieldErrors({'email':AJS.I18n.getText('bitbucket.web.setup.email.error')}));
                return;
            }

            setRadioAndSubmitButtonsDisabled(true);
            showSpinner($el, AJS.I18n.getText('bitbucket.web.setup.evaluate.description.step1.spinner'));

            var url = lassoUrl + 'profile/rest/user/' + data.email;

            var emailRequest = performAjaxRequest({
                'url': url,
                'type': 'GET',
                'jsonpType': 'GET'
            });

            emailRequest.done(function(response) {
                setRadioAndSubmitButtonsDisabled(false);
                hideSpinner();
                if (response.exists) {
                    showLogin(data);
                } else {
                    showSignup(data);
                }
            });
        });

        // When the submit button is clicked in the no-account form a new MAC user needs to be created
        $document.on('submit', '#no-account', function(e) {
            e.preventDefault();
            var $el = $(e.currentTarget);

            var accountDetails = extractFormData({
                "displayName": 'full-name',
                "email": 'no-account-email',
                "password": 'no-account-password',
                "confirmPassword": 'confirm-password',
                "companyName": 'company-name',
                "country": 'country'
            });
            accountDetails.country = $("#country").val();
            accountDetails.lastName = '';

            // Make sure all required fields are filled in
            if (checkRequiredFields($el, accountDetails, bitbucket.internal.page.setup.noAccount)) {
                makeCountrySelectorAui(accountDetails.country);
                return;
            }

            setRadioAndSubmitButtonsDisabled(true);
            showSpinner($el, AJS.I18n.getText('bitbucket.web.setup.account.signup'));
            createMACAccount(accountDetails, $el);
        });

        // When the submit button is clicked in the no-key form a license key needs to be generated for the given user
        $document.on('submit', '#no-key', function(e) {
            e.preventDefault();
            var $el = $(e.currentTarget);

            var accountDetails = extractFormData({
                'username': 'no-key-email',
                'password': 'no-key-password'
            });

            // Make sure all required fields are filled in
            if (checkRequiredFields($el, accountDetails, bitbucket.internal.page.setup.noKey)) {
                return;
            }

            setRadioAndSubmitButtonsDisabled(true);
            showSpinner($el, AJS.I18n.getText('bitbucket.web.setup.account.signin'));
            signin(accountDetails, $el);
        });

        // When the next button is clicked on the show license key form, submit the entire form
        $document.on('submit', '#license-complete', function(e) {
            e.preventDefault();
            var $el = $(e.currentTarget);
            setRadioAndSubmitButtonsDisabled(true);
            showSpinner($el, AJS.I18n.getText('bitbucket.web.setup.license.import', bitbucket.internal.util.productName()));
            $settingsForm.submit();
        });

        // When the submit button is clicked in the has-key form, just fill the license key form
        $document.on('submit', '#has-key', function(e) {
            e.preventDefault();
            var $el = $(e.currentTarget);

            var licenseKey = $('#license').val();
            // Make sure all required fields are filled in
            if (checkRequiredFields($el, {'license':licenseKey}, bitbucket.internal.page.setup.hasKey)) {
                return;
            }

            setRadioAndSubmitButtonsDisabled(true);
            showSpinner($el, AJS.I18n.getText('bitbucket.web.setup.license.import', bitbucket.internal.util.productName()));
            loadLicense(licenseKey, "hasKey");
            $settingsForm.submit();
        });
    };

    /**
     * Show the login form
     * @param {Object<string, string>} data
     */
    function showLogin(data) {
        $licenseForm.hide();
        $("#no-key").show();
        $("#no-key-email").val(data.email);
        $("#no-key-password").focus();
        setupTracking.track('setup-settings-signIn');
    }

    /**
     * Show the signUp form
     * @param {Object<string, string>} data
     */
    function showSignup(data) {
        $(".license-form").hide();
        $("#no-account").show();
        makeCountrySelectorAui(data.country);
        $("#no-account-email").val(data.email);
        $("#full-name").focus();
        setupTracking.track('setup-settings-signUp');
    }

    /**
     * Make a call to Lasso to sign the user in
     * @param {Object<string, string>} accountDetails
     * @param {jQuery} $form
     */
    function signin(accountDetails, $form) {
        performAjaxRequest({
            'url': lassoUrl + "id/rest/login",
            'type': 'POST',
            'jsonpType': 'POST',
            'data': accountDetails
        }).done(function(responseJson) {
            generateEvaluationLicense(responseJson.xsrfToken, $("#server-id").text(), "signIn", $form);
        }).fail(function(xhr) {
            loadSoyTemplate($form, accountDetails, bitbucket.internal.page.setup.noKey, formatFormErrors(xhr.responseJSON.error), formatFieldErrors(xhr.responseJSON.fieldErrors));
            setRadioAndSubmitButtonsDisabled(false);
            return false;
        });
    }

    /**
     * Make a call to Lasso to create a new user
     * @param {Object<string, string>} accountDetails
     * @param {jQuery} $form
     */
    function createMACAccount(accountDetails, $form) {
        performAjaxRequest({
            'url': lassoUrl + "profile/rest/signUp",
            'type': 'POST',
            'jsonpType': 'GET',
            'data': accountDetails
        }).done(function(responseJson) {
            generateEvaluationLicense(responseJson.xsrfToken, $("#server-id").text(), "signUp", $form);
        }).fail(function(xhr) {
            loadSoyTemplate($form, accountDetails, bitbucket.internal.page.setup.noAccount, formatFormErrors(xhr.responseJSON.error), formatFieldErrors(xhr.responseJSON.fieldErrors));
            makeCountrySelectorAui(accountDetails.country);
            setRadioAndSubmitButtonsDisabled(false);
            return false;
        });
    }

    /**
     *
     * @param {string} xsrfToken
     * @param {number} serverId
     * @param {string} from
     * @param {jQuery} $form
     */
    function generateEvaluationLicense(xsrfToken, serverId, from, $form) {
        showSpinner($form, AJS.I18n.getText('bitbucket.web.setup.license.generate'));
        var data = {
            productKey: "stash",
            serverId: serverId
        };
        // Get an evaluation license from HAMS
        performAjaxRequest({
            'url': hamletUrl + "1.0/public/license/createEvaluation",
            'type': 'POST',
            'jsonpType': 'GET',
            'beforeSend': function(xhr) {
                xhr.setRequestHeader("ATL-XSRF-Token", xsrfToken);
            },
            'xsrfToken': xsrfToken,
            'data': data
        }).done(function(responseJson) {
            loadAndShowLicense(responseJson.licenseKey, from);
        }).fail(function(xhr) {
            hideSpinner();
            data.from = from;
            loadSoyTemplate($("#has-key"), data, bitbucket.internal.page.setup.hasKey, formatFormErrors(xhr.responseJSON.error), formatFieldErrors(xhr.responseJSON.fieldErrors));
            setRadioAndSubmitButtonsDisabled(false);
            return false;
        });
    }

    /**
     * Copy the license key across to the hidden license key field to be submitted by the main form
     * @param {string} licenseKey
     */
    function loadLicense(licenseKey) {
        $("#licenseKeyHidden").val(licenseKey);
    }

    /**
     * Prepare the page to show and submit the license key to Stash
     * @param {string} licenseKey
     * @param {string} from
     */
    function loadAndShowLicense(licenseKey, from) {
        loadLicense(licenseKey);
        loadSoyTemplate($("#license-complete"), {'licenseKey': licenseKey, 'from': from}, bitbucket.internal.page.setup.licenseGenerationComplete, null, null);
        setupTracking.track('setup-settings-eval-generate-success');
    }

    /**
     * Perform a generic ajax request
     * @param {Object<string, string>} options
     * @returns {Promise} Request object
     */
    function performAjaxRequest(options) {
        var request;
        options.timeout = 30000;

        if (browserSupportsCORS()) {
            request = $.ajax({
                url: options.url,
                type: options.type,
                beforeSend: options.beforeSend,
                xhrFields: {
                    withCredentials: true
                },
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify(options.data),
                timeout: options.timeout
            });
            request.fail(function(xhr) {
                genericError();
            });
        } else {
            genericError(AJS.I18n.getText('bitbucket.web.setup.error.outdated',
                AJS.escapeHtml($("#generateEvalUrlHidden").val())), options.initialValues);
            request = $.Deferred();
        }
        return request;
    }

    /**
     * Transform the field errors from MAC or HAMS to be in the format that the soy expects
     * @param {Object<string, string>} fieldErrors
     * @returns {Object<string, Array>}
     */
    function formatFieldErrors(fieldErrors) {
        // The soy form expects field errors to be mapped to lists, but the response is a string.
        if (!fieldErrors) {
            return null;
        }
        return _.mapValues(fieldErrors, _.ary(Array, 1));
    }

    /**
     * Transform the form errors from MAC or HAMS to be in the format that the soy expects
     * @param {Object<string, string>} formErrors
     * @returns {Object<string, Array>}
     */
    function formatFormErrors(formErrors) {
        // The soy form expects field errors to be mapped to lists, but the response is sometimes a string.
        if (!formErrors) {
            return null;
        }
        return [].concat(formErrors);
    }

    /**
     * Re-render the soy template for a certain function, passing parameters into it
     * @param {jQuery} $form
     * @param {Object<string, string>} initialValues
     * @param {Function} soyFunction
     * @param {Object<string, Array>} formErrors
     * @param {Object<string, Array>} fieldErrors
     */
    function loadSoyTemplate($form, initialValues, soyFunction, formErrors, fieldErrors) {
        if (initialValues == null) {
            initialValues = {};
        }

        if (!initialValues.serverId) {
            initialValues.serverId = $("#server-id").text();
        }
        if (!initialValues.productKey) {
            initialValues.productKey = "stash";
        }

        var soyParams = {};
        soyParams.initialValues = initialValues;

        var xsrfTokenName = "atl_token";
        soyParams.xsrfTokenName = xsrfTokenName;
        // Can't use extractFieldValue() because it is extracted based on name, not ID
        var xsrfTokenValue = $form.find('input[name="' + xsrfTokenName + '"]').val();
        soyParams.xsrfTokenValue = xsrfTokenValue;

        if (formErrors) {
            soyParams.generateEvalUrl = $("#generateEvalUrlHidden").val();
            soyParams.formErrors = formErrors;
        }

        if (fieldErrors) {
            soyParams.fieldErrors = fieldErrors;
        }

        // Replace the current form with the new form that has errors and insert the user's input
        $licenseForm.hide();
        var $newForm = $(soyFunction(soyParams));
        $form.replaceWith($newForm);
        $licenseForm = $('.license-form');
        $newForm.show(); // The form is hidden by default, but it needs to be shown for the errors
        // Find the first field with an error and focus on that
        var fields = $newForm.find("input,textarea");
        if (fieldErrors) {
            _.first(_.filter(fields, function(field) {
                var key;
                if (field.name) {
                    key = field.name;
                } else {
                    key = field.id;
                }
                return Object.keys(fieldErrors).indexOf(key) > -1;
            })).focus();
        }

        if ($form.attr("id") === "has-key") {
            document.getElementById('has-key-radio').checked = true;
        } else {
            document.getElementById('check-email-radio').checked = true;
        }

        // Reset the spinners
        hideSpinner();
    }

    /**
     * Retrieve data from a form as a map
     * @param {Object<string, string>} fieldMap - map of desired key to field ID (key -> id)
     * @returns {Object<string, string>}
     */
    function extractFormData(fieldMap) {
        var data = {};
        for (var fieldId in fieldMap) {
            if (fieldMap.hasOwnProperty(fieldId)) {
                data[fieldId] = document.getElementById(fieldMap[fieldId]).value;
            }
        }
        return data;
    }

    /**
     * Client side check of required fields to make sure they are not empty
     * @param {jQuery} $form
     * @param {Object<string, string>} data
     * @param {Function} soyTemplate
     * @returns {boolean}
     */
    function checkRequiredFields($form, data, soyTemplate) {
        var errorMap = {};

        $form.find('input.required,textarea.required').each(function(i, element) {
            if (element.value === '') {
                var key;
                if (element.name) {
                    key = element.name;
                } else {
                    key = element.id;
                }
                errorMap[key] = AJS.I18n.getText('bitbucket.web.setup.error.required');
            }
        });

        if (Object.keys(errorMap).length === 0) {
            return false;
        } else {
            loadSoyTemplate($form, data, soyTemplate, null, formatFieldErrors(errorMap));
            return true;
        }
    }

    /**
     * Initialise the country selector
     * @param {string} selectedValue
     */
    function makeCountrySelectorAui(selectedValue) {
        var $country = $("#country").html("");
        $country.auiSelect2();
        if (!countries) {
            updateCountries();
        }
        $country.html(countries);
        $country.auiSelect2('val', selectedValue);
    }

    /**
     * For non-specialised errors, provide this default error
     * @returns {boolean} false
     */
    function genericError(messageHtml, initialValues) {
        if (!messageHtml || typeof(messageHtml) !== 'string') {
            messageHtml = AJS.I18n.getText('bitbucket.web.setup.error');
        }
        /* globals soydata */
        loadSoyTemplate($("#has-key"), initialValues, bitbucket.internal.page.setup.hasKey, [soydata.VERY_UNSAFE.ordainSanitizedHtml(messageHtml)], null);
        setRadioAndSubmitButtonsDisabled(false);
        return false;
    }

    /**
     * Retrieves the country codes from HAMS
     * @returns {Promise} country request
     */
    function getCountryCodes() {
        var url = hamletUrl + "1.0/public/country/all";
        return performAjaxRequest({
            type: 'GET',
            jsonpType: 'GET',
            url: url,
            initialValues: {
                license: $("#license").val()
            }
        });
    }

    /**
     * Show the spinner for a given form
     * @param {jQuery} $form
     * @param {string} msg
     */
    function showSpinner($form, msg) {
        $form.find('.button-spinner').show().spin();
        $form.find('.next-text').show().text(msg);
    }

    /**
     * Hide all the spinners
     */
    function hideSpinner() {
        $('.button-spinner').spinStop().hide();
        $('.next-text').hide();
    }

    /**
     * Disable/enable the radio and submit buttons (Used while waiting for responses etc)
     * @param {boolean} disabled
     */
    function setRadioAndSubmitButtonsDisabled(disabled) {
        $('input[type="radio"], input[type="submit"]').prop('disabled', disabled);
    }

    /**
     * Retrieve a country list from HAMS and store in global variable countries
     */
    function updateCountries() {
        countries = "";
        getCountryCodes().done(function (countryJson) {
            var countryList = countryJson.countryList;
            countryList.sort(function (a, b) {
                return a.displayName.localeCompare(b.displayName);
            });
            countryList.forEach(function(country) {
                countries += '<option value=' + country.countryIsoCode + '>' + country.displayName + '</option>';

            });
        });
    }
});
