define('bitbucket/internal/page/source', [
    'aui',
    'jquery',
    'lodash',
    'memoir',
    'bitbucket/util/navbuilder',
    'bitbucket/internal/feature/commit/commit-badge',
    'bitbucket/internal/feature/file-content',
    'bitbucket/internal/layout/page-scrolling-manager',
    'bitbucket/internal/model/commit-range',
    'bitbucket/internal/model/file-change',
    'bitbucket/internal/model/file-content-modes',
    'bitbucket/internal/model/page-state',
    'bitbucket/internal/model/path',
    'bitbucket/internal/model/revision',
    'bitbucket/internal/model/revision-reference',
    'bitbucket/internal/util/ajax',
    'bitbucket/internal/util/events',
    'bitbucket/internal/util/function',
    'exports'
], function(
    AJS,
    $,
    _,
    memoir,
    nav,
    commitBadge,
    FileContent,
    pageScrollingManager,
    CommitRange,
    FileChange,
    FileContentModes,
    pageState,
    Path,
    Revision,
    RevisionReference,
    ajax,
    events,
    fn,
    exports
) {

    var dialogIsShowing;

    exports.onReady = function(path, atRevisionRef, untilRevision, mode, fileContentContainerSelector, fileContentId, relevantContextLines) {

        pageScrollingManager.acceptScrollForwardingRequests();

        var currentUrl = window.location.href;
        var currentPath = new Path(path);
        var currentHeadRef = new RevisionReference(atRevisionRef);
        var currentUntilRevision = new Revision(untilRevision);
        var currentMode = FileContentModes.DIFF === mode ? FileContentModes.DIFF : FileContentModes.SOURCE;

        var fileContent = new FileContent(fileContentContainerSelector, fileContentId, FileContent.sourcePreset);

        events.on('memoir.changestate', function(e) {

            var newPath;
            var newHeadRef;
            var newUntilRevision;
            var newMode;

            var state = e.state;
            if (state) {
                newPath = new Path(state.path);
                newHeadRef = new RevisionReference(state.headRef);
                newUntilRevision = state.untilRevision ? new Revision(state.untilRevision) : null;
                newMode = state.mode ? state.mode : FileContentModes.SOURCE;

                var currentUntilId = currentUntilRevision ? currentUntilRevision.getId() : null;
                var newUntilId = newUntilRevision ? newUntilRevision.getId() : null;

                var pathChanged = newPath.toString() !== currentPath.toString();
                var headRefChanged = currentHeadRef.getId() !== newHeadRef.getId();
                var untilRevisionChanged = newUntilId !== currentUntilId;
                var modeChanged = newMode !== currentMode;
                var stateChanged = pathChanged || headRefChanged || untilRevisionChanged || modeChanged;

                currentPath = newPath;
                currentUntilRevision = newUntilRevision;
                currentHeadRef = newHeadRef;
                currentMode = newMode;

                if (headRefChanged) {
                    events.trigger('bitbucket.internal.page.source.revisionRefChanged', null, currentHeadRef);
                }

                // it's possible this we're just popping a hashchange. Check that state actually changed.
                if (stateChanged) {
                    updateForState();
                }
                currentUrl = window.location.href;
            } else { // this should just be a hash change, so it should be ignorable.
                // however, browsers don't always persist state data
                // see: https://github.com/balupton/history.js/wiki/The-State-of-the-HTML5-History-API
                // "State persisted when navigated away and back"
                // in that case, we have to either regrab all the state (path from url, headRef and untilRevision from ??)
                // or reload the page. Reloading the page because it's the easier option.

                var isHashChangeOnly = _.endsWith(urlWithoutHash(window.location.href), urlWithoutHash(currentUrl));

                if (!isHashChangeOnly) {
                    window.location.reload();
                }
            }
        });

        function getCurrentCommitRange() {
            return new CommitRange({
                untilRevision : currentUntilRevision,
                sinceRevision : currentUntilRevision.hasParents() ?
                    currentUntilRevision.getParents()[0] :
                    undefined
            });
        }

        function urlWithoutHash(url) {
            var hashIndex = url.lastIndexOf('#');
            return hashIndex === -1 ? url : url.substring(0, hashIndex);
        }

        // Trigger a state change to refresh the file currently shown in the diff view.
        // Use case: diff options have changed and a new representation of the file needs to be shown.
        events.on('bitbucket.internal.feature.fileContent.optionsChanged', function(change) {
            var nonRefreshKeys = ['hideComments', 'hideEdiff'];

            if(!_.contains(nonRefreshKeys, change.key)) {
                updateForState();
            }
        });


        var pendingRequest = null;
        function updateForState() {

            // if we're still updating from a previous request, abort that update.
            if (pendingRequest) {
                pendingRequest.abort();
                pendingRequest = null;
            }

            // headRef must've changed, so we don't actually know the untilRevision anymore
            if (!currentUntilRevision) {
                // find the latest commit where this file was changed.  This might not be the headRef itself.
                pendingRequest = getLatestFileRevision(currentPath, currentHeadRef);

                fileContent.reset(); // Destroy the view

                pendingRequest.always(function() {
                    pendingRequest = null;
                }).done(function(revision) {
                    currentUntilRevision = revision;
                    updateForState();
                });
            } else {
                initFileContent();
                updateCommitBadge(currentUntilRevision);
            }
        }

        function initFileContent() {
            var options = $.extend({
                toolbarWebFragmentLocationPrimary: 'bitbucket.file-content.'+ currentMode +'.toolbar.primary',
                toolbarWebFragmentLocationSecondary: 'bitbucket.file-content.'+ currentMode +'.toolbar.secondary'
            },
            FileContent[ currentMode + 'Preset'], {
                relevantContextLines: relevantContextLines,
                headRef: currentHeadRef,
                anchor: window.location.hash.substr(1) || undefined
            });

            var fileChange = new FileChange({
                commitRange: getCurrentCommitRange(),
                path: currentPath,
                repository: pageState.getRepository()
            });

            return fileContent.init(fileChange, options);
        }

        function updateCommitBadge(untilRevision) {
            $('.branch-selector-toolbar .commit-badge-container').empty().append(
                commitBadge.create(untilRevision.toJSON(), pageState.getRepository().toJSON())
            ).fadeIn('fast');
        }

        function stateObject(path, headRef, untilRevision, mode) {
            return {
                path : path.toString(),
                headRef : headRef.toJSON(),
                untilRevision : untilRevision ? untilRevision.toJSON() : null,
                mode : mode
            };
        }

        function pushState(path, headRef, untilRevision, mode) {
            var urlBuilder = nav.currentRepo();
            if (mode === FileContentModes.DIFF) {
                urlBuilder = urlBuilder.diff(
                    new FileChange({
                        commitRange: new CommitRange({
                            untilRevision : untilRevision // Since is the revision's parent but not needed in the URL
                        }),
                        path : path,
                        repository : pageState.getRepository()
                    })
                );
            } else {
                urlBuilder = urlBuilder.browse().path(path);
                if (untilRevision) {
                    urlBuilder = urlBuilder.until(untilRevision.getId());
                }
            }

            if (!headRef.isDefault()) {
                urlBuilder = urlBuilder.at(headRef.getId());
            }
            memoir.pushState(stateObject(path, headRef, untilRevision, mode), null, urlBuilder.build());
        }

        function getLatestFileRevision(path, revisionReference) {

            var urlBuilder = nav.rest()
                .currentRepo()
                .commit(revisionReference.getLatestCommit());

            var xhr = ajax.rest({
                url : urlBuilder.withParams({
                    path: path.toString(),
                    avatarSize: bitbucket.internal.widget.avatarSizeInPx({ size: 'xsmall' })
                }).build(),
                statusCode: {
                    '404': function() {
                        //HACK: this fallback shouldn't be necessary. We should always get a value.
                        // Actually, this case should happen when the revision doesn't exist on the branch you select.
                        // We should handle it differently.

                        return $.Deferred().resolve({
                            id : revisionReference.getId(),
                            displayId : revisionReference.getDisplayId(),
                            author : {"name": "Unknown"},
                            authorTimestamp : NaN
                        });
                    }
                }
            });
            var pipedPromise = xhr.then(function(latestCommit) {
                    return new Revision(latestCommit);
                });

            events.trigger('bitbucket.internal.page.source.requestedRevisionData');
            return pipedPromise.promise(xhr);
        }

        memoir.initialState(stateObject(currentPath, currentHeadRef, currentUntilRevision, currentMode));

        initFileContent();

        events.on('bitbucket.internal.layout.branch.revisionRefChanged', function(revisionReference) {
            if (currentHeadRef !== revisionReference) {
                // the new commit reference isn't necessarily the commit on which the file was changed.
                // we must find the latest one where it was changed. hence why untilRevision is null
                // Always revert back to source view - doesn't make sense to keep on diff view when switching branches.
                pushState(currentPath, revisionReference, null, "source");
            }
        });

        events.on('bitbucket.internal.feature.*.untilRevisionChanged', function(revision) {
            if (currentUntilRevision.getId() !== revision.getId()) {
                pushState(currentPath, currentHeadRef, revision, currentMode);
            }
        });

        events.on('bitbucket.internal.feature.*.requestedModeChange', function(mode) {
            if (currentMode !== mode) {
                pushState(currentPath, currentHeadRef, currentUntilRevision, mode);
            }
        });

        events.on('bitbucket.internal.feature.sourceview.onError', function(errors) {
            $('.branch-selector-toolbar .commit-badge-container').fadeOut('fast');
        });

        events.on('bitbucket.internal.layout.*.urlChanged', function(url) {
            window.location = url;
            //TODO: pushState back to fileBrowser
            //events.trigger('bitbucket.internal.page.source.urlChanged', null, url);
        });
        events.on('bitbucket.internal.feature.*.urlChanged', function(url) {
            window.location = url;
            //TODO: pushState back to fileBrowser
            //events.trigger('bitbucket.internal.page.source.urlChanged', null, url);
        });

        events.on('bitbucket.internal.widget.branchselector.dialogShown', function () {
            dialogIsShowing = true;
        });
        events.on('bitbucket.internal.widget.branchselector.dialogHidden', function () {
            dialogIsShowing = false;
        });

        $(window).on('hashchange', function () {
            currentUrl = window.location.href;

            if (memoir.nativeSupport()) {
                memoir.replaceState(stateObject(currentPath, currentHeadRef, currentUntilRevision), null, currentUrl);
            }

            var anchor = window.location.hash.substr(1) || undefined;

            //Quack quack
            fn.dotX('handler.updateAnchor', anchor)(fileContent);

            events.trigger('bitbucket.internal.page.source.anchorChanged', null, anchor);
        });

        events.on('bitbucket.internal.widget.keyboard-shortcuts.register-contexts', function(keyboardShortcuts) {
            keyboardShortcuts.enableContext('sourceview');
            keyboardShortcuts.enableContext('diff-view');
        });

        listenForKeyboardShortcutRequests();
    };

    function listenForKeyboardShortcutRequests() {
        events.on('bitbucket.internal.keyboard.shortcuts.requestOpenParentHandler', function(keys) {
            (this.execute ? this : AJS.whenIType(keys)).execute(function() {
                if (!dialogIsShowing) {
                    var $parentDir = $('.breadcrumbs').find('a:last');
                    if ($parentDir.length) {
                        if (memoir.nativeSupport()) {
                            $parentDir.click();
                        } else {
                            window.location.href = $parentDir.attr('href');
                        }
                    }
                }
            });
        });
    }
});
