define('bitbucket/internal/widget/sidebar', [
    'aui',
    'chaperone',
    'jquery',
    'bitbucket/internal/util/client-storage',
    'bitbucket/internal/util/events',
    'exports'
], function(
    AJS,
    Chaperone,
    $,
    clientStorage,
    events,
    exports
) {

    'use strict';

    var IS_EXPANDED_KEY = 'sidebar_expanded';

    var sidebarSelector = '.aui-sidebar';
    var sidebar;

    function initSidebar() {
        sidebar = AJS.sidebar(sidebarSelector);

        sidebar.on('collapse-end', function(e) {
            if (!e.isResponsive) {
                clientStorage.setItem(IS_EXPANDED_KEY, false);
            }
            events.trigger('bitbucket.internal.feature.sidebar.collapseEnd');

        });
        sidebar.on('expand-end', function(e) {
            if (!e.isResponsive) {
                clientStorage.setItem(IS_EXPANDED_KEY, true);
            }
            events.trigger('bitbucket.internal.feature.sidebar.expandEnd');
        });

        initFeatureDiscovery();

        initSidebarUIEvents();
    }

    function initFeatureDiscovery() {
        var isExpanded = !isCollapsed();
        sidebar.on('expand', function() { isExpanded = true; });
        sidebar.on('collapse', function() { isExpanded = false; });

        Chaperone.registerFeature('sidebar', [{
            id: 'sidebar-discovery',
            selector: '.aui-sidebar[aria-expanded=false] .aui-page-header, .aui-sidebar[aria-expanded=true] .aui-sidebar-group:first-child',
            title: AJS.I18n.getText('stash.web.global.sidebar.feature.discovery.title', bitbucket.internal.util.productName()),
            content: bitbucket.internal.widget.sidebar.discoveryContent(),
            moreInfo: {
                href: 'http://go.atlassian.com/stashsidebar'
            },
            gravity: 'w',
            width: 350,
            offsetX: function() { return isExpanded ? -10 : 10; },
            once: true
        }]);

        sidebar.on('expand', Chaperone.checkFeatureAlignment);
        sidebar.on('collapse', Chaperone.checkFeatureAlignment);
        sidebar.on('expand-end', Chaperone.checkFeatureAlignment);
        sidebar.on('collapse-end', Chaperone.checkFeatureAlignment);

    }

    function preloadSidebar() {
        var state = isCollapsed();
        $(document.body).toggleClass("aui-sidebar-collapsed", state);
        $(sidebarSelector).attr("aria-expanded", !state);
    }

    /**
     * Is the sidebar currently collapsed
     *
     * @returns {boolean}
     */
    function isCollapsed() {
        return !clientStorage.getItem(IS_EXPANDED_KEY);
    }

    /**
     * Initialise the sidebar UI events that will trigger bitbucket.internal.ui.* events
     */
    function initSidebarUIEvents() {
        // Hover on the actions menu trigger when the sidebar is collapsed
        // When STASHDEV-8893 and ACCESS-384 are resolved, look at also including appropriate listeners
        // for keyboard focus which triggers the menu to open.
        sidebar.$el.on('mouseenter', '.aui-sidebar-group-actions', function(e) {
            if (isCollapsed()) {
                events.trigger('bitbucket.internal.ui.sidebar.actions-menu.open');
            }
        });

        // Click on any items in the actions menu
        sidebar.$el.find('.aui-sidebar-group-actions ul').on('click', '> li > a[data-web-item-key]', function() {
            events.trigger('bitbucket.internal.ui.sidebar.actions-menu.item.clicked', null, {
                isCollapsed: isCollapsed(),
                webItemId: $(this).attr('data-web-item-key')
            });
        });

        // Click on any navigation items
        sidebar.$el.find('.sidebar-navigation ul').on('click', '> li > a[data-web-item-key]', function() {
            var $el = $(this);
            var isCollapsed = !clientStorage.getItem(IS_EXPANDED_KEY);
            var listLevel = $el.parentsUntil('.aui-sidebar-group').filter('ul').length;

            events.trigger('bitbucket.internal.ui.sidebar.item.clicked', null, {
                webItemId: $el.attr('data-web-item-key'),
                isCollapsed: isCollapsed,
                level: listLevel
            });
        });

        // Click on the settings button
        sidebar.$el.find('.sidebar-settings-group').on('click', 'a', function() {
            events.trigger('bitbucket.internal.ui.sidebar.settings.clicked', null, {
                webItemId: $(this).attr('data-web-item-key')
            });
        });


        // Monitor the toggling of expand/collapse.
        // These are the same event handlers and selectors/filters used in aui-sidebar
        sidebar.$el.on('click', '.aui-sidebar-toggle', function(e) {
            triggerCollapseChange('button');
        });

        sidebar.$el.on('click', '.aui-sidebar-body', function(e) {
            if ($(e.target).is('.aui-sidebar-body')) {
                triggerCollapseChange('sidebar');
            }
        });

        AJS.whenIType('[').execute(function() {
            triggerCollapseChange('keyboard-shortcut');
        });

        sidebar.on('expand-end collapse-end', function(e) {
            if (e.isResponsive) {
                triggerCollapseChange('resize');
            }
        });

    }

    /**
     * Trigger a collapse state changed ui event
     *
     * @param {string} source
     */
    function triggerCollapseChange(source) {
        events.trigger('bitbucket.internal.ui.sidebar.collapse.change', null, {
            source: source,
            isCollapsed: isCollapsed(),
            windowWidth: window.innerWidth
        });
    }

    exports.preload = preloadSidebar;
    exports.onReady = initSidebar;
});
