package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

public class MergeException extends ScmException {

    private final boolean conflicted;
    private final String from;
    private final Repository repository;
    private final String to;

    /**
     * @param message message
     * @param scmId ID of the scm
     * @param repository repository
     * @param from from hash
     * @param to to hash
     * @param conflicted true if conflicted
     * @since 5.0
     */
    public MergeException(@Nonnull KeyedMessage message, @Nonnull String scmId,
                          @Nonnull Repository repository, @Nonnull String from, @Nonnull String to,
                          boolean conflicted) {
        this(message, null, scmId, repository, from, to, conflicted);
    }

    /**
     * @param message message
     * @param cause cause or null
     * @param scmId ID of the scm
     * @param repository repository
     * @param from from hash
     * @param to to hash
     * @param conflicted true if conflicted
     * @since 5.0
     */
    public MergeException(@Nonnull KeyedMessage message, @Nullable Throwable cause, @Nonnull String scmId,
                          @Nonnull Repository repository, @Nonnull String from, @Nonnull String to,
                          boolean conflicted) {
        super(message, cause, scmId);

        this.conflicted = conflicted;
        this.repository = requireNonNull(repository, "repository");
        this.from = requireNonNull(from, "from");
        this.to = requireNonNull(to, "to");
    }

    @Nonnull
    public String getFrom() {
        return from;
    }

    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Nonnull
    public String getTo() {
        return to;
    }

    /**
     * @return {@code true} if the merge failed because the resulting merge had conflicts
     * @since 5.0
     */
    public boolean isConflicted() {
        return conflicted;
    }
}
