package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.i18n.KeyedMessage;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * A specialization of {@link CommandFailedException} thrown when a command fails because it is called incorrectly.
 * <p>
 * This exception is an indication of a <i>developer error</i>, not a <i>user error</i>. In practice, these exceptions
 * should never occur during normal operation.
 * <p>
 * Developer note: When considering the {@link #getCommand() command} against the {@link #getUsage() usage} message, be
 * aware that representing the command as a single {@code String} may conceal subtle usage issues. For example, when
 * providing multiple arguments to a {@link CommandBuilder command builder} via its
 * {@link CommandBuilder#argument(String) argument(String)}, providing multiple arguments in a
 * single call separated by spaces will result in the arguments not being received correctly by the command when it is
 * executed but will produce the same {@code String} representation. For example, both of the following will produce the
 * same {@code String}:
 * <pre><code>
 *     builder.command("command").argument("a b").build(...).call();
 *
 *     builder.command("command").argument("a").argument("b").build(...).call();
 * </code></pre>
 * However, the executed command will receive different arguments ({@code "a b"} and {@code ["a", "b"]}, respectively).
 * If the command arguments appear to be valid based on the usage message, double-check how they are provided.
 */
public class CommandUsageException extends CommandFailedException {

    private final String command;
    private final String usage;

    public CommandUsageException(@Nonnull KeyedMessage message, @Nonnull String command, @Nonnull String usage) {
        super(message);

        this.command = requireNonNull(command, "command");
        this.usage = requireNonNull(usage, "usage");
    }

    /**
     * @return the command as it was executed
     */
    @Nonnull
    public String getCommand() {
        return command;
    }

    /**
     * @return the usage message
     */
    @Nonnull
    public String getUsage() {
        return usage;
    }
}
