package com.atlassian.bitbucket.pull.reviewer;

import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;

/**
 * @since 7.13
 */
public interface ReviewerGroupService {

    /**
     * Create a new reviewer group
     *
     * @param request describes the reviewer group to create
     * @return the created {@link ReviewerGroup reviewer group}
     * @throws EmptyReviewerGroupException when creating a group with no users
     */
    @Nonnull
    ReviewerGroup create(@Nonnull ReviewerGroupCreateRequest request);

    /**
     * Delete a reviewer group with the given ID.
     *
     * @param reviewerGroupId the ID of the {@link ReviewerGroup reviewer group} to be deleted
     * @throws NoSuchReviewerGroupException if the ID of the reviewer group to be deleted does not exist
     */
    void delete(long reviewerGroupId);

    /**
     * Find reviewer groups by name.
     * <p>
     * Will also find all the reviewer groups which begin with the name or partial name provided within a given scope.
     *
     * @param scope the {@link Scope scope} to retrieve the reviewer groups from
     * @param name the name or start of the name of a reviewer group
     * @param pageRequest the page of reviewer groups to be returned
     * @return a page containing 0 or more {@link ReviewerGroup reviewer groups} that has the name provided or starts
     * with the partial name provided.
     */
    @Nonnull
    Page<ReviewerGroup> findByName(@Nonnull Scope scope, @Nonnull String name, @Nonnull PageRequest pageRequest);

    /**
     * Find the reviewer groups in a given scope
     *
     * @param scope the {@link Scope scope} to retrieve the reviewer groups from
     * @param pageRequest the page of reviewer groups to be returned
     * @return a page containing 0 or more {@link ReviewerGroup reviewer groups}
     */
    @Nonnull
    Page<ReviewerGroup> findByScope(@Nonnull Scope scope, @Nonnull PageRequest pageRequest);

    /**
     * Find a reviewer group with the given ID.
     * <p>
     * If no reviewer group exists with the specific ID, an exception is thrown.
     *
     * @param reviewerGroupId the ID of the reviewer group to be retrieved
     * @return the {@link ReviewerGroup reviewer group} with the specified ID
     * @throws NoSuchReviewerGroupException if no reviewer group exists with the specific ID
     */
    @Nonnull
    ReviewerGroup getById(long reviewerGroupId);

    /**
     * Updates a reviewer group.
     * <p>
     * Only the name, description and users of the group can be updated. The name and users of a group can never be {@code null},
     * however the users list can be empty. A user who already exists in the group will not be added to prevent duplicates.
     * The {@link Scope scope} of a reviewer group cannot be changed after creation.
     *
     * @param request a request containing the attributes to update for the reviewer group
     * @return the updated {@link ReviewerGroup reviewer group}
     * @throws NoSuchReviewerGroupException if the ID of the reviewer group to be updated does not exist
     */
    @Nonnull
    ReviewerGroup update(@Nonnull ReviewerGroupUpdateRequest request);
}
