package com.atlassian.bitbucket.pull;

import javax.annotation.Nonnull;
import java.util.function.Predicate;

/**
 * Enumerates the possible {@link PullRequestParticipant#getRole() roles} a participant can have in a pull request.
 *
 * @see PullRequestParticipant
 * @see PullRequest#getAuthor()
 * @see PullRequest#getParticipants()
 * @see PullRequest#getReviewers()
 */
public enum PullRequestRole {

    /**
     * The {@link PullRequestParticipant participant} is an author of the {@link PullRequest pull request}.
     */
    AUTHOR(0, true),

    /**
     * The {@link PullRequestParticipant participant} is a reviewer of the {@link PullRequest pull request}.
     */
    REVIEWER(1, true),

    /**
     * The {@link PullRequestParticipant participant} has no explicit role in the {@link PullRequest pull request}.
     */
    PARTICIPANT(2, false);

    private final int id;
    private final boolean explicit;

    PullRequestRole(int id, boolean explicit) {
        this.id = id;
        this.explicit = explicit;
    }

    @Nonnull
    public static PullRequestRole fromId(int id) {
        for (PullRequestRole value : values()) {
            if (value.getId() == id) {
                return value;
            }
        }
        throw new IllegalArgumentException("No PullRequestRole is associated with ID [" + id + "]");
    }

    @Nonnull
    public Predicate<PullRequestParticipant> getFilter() {
        return participant -> participant.getRole() == PullRequestRole.this;
    }

    public int getId() {
        return id;
    }

    public boolean isExplicit() {
        return explicit;
    }
}
