package com.atlassian.bitbucket.migration;

import javax.annotation.Nonnull;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Represents a request to export a set of repositories.
 *
 * @since 5.13
 */
public class ExportRequest {

    private final String exportLocation;
    private final RepositoriesExportRequest repositoriesRequest;

    private ExportRequest(@Nonnull Builder builder) {
        exportLocation = builder.exportLocation;
        repositoriesRequest = builder.repositoriesRequest;
    }

    @Nonnull
    public Optional<String> getExportLocation() {
        return ofNullable(exportLocation);
    }

    @Nonnull
    public RepositoriesExportRequest getRepositoriesRequest() {
        return repositoriesRequest;
    }

    public static class Builder {

        private String exportLocation;
        private RepositoriesExportRequest repositoriesRequest;

        @Nonnull
        public ExportRequest build() {
            if (repositoriesRequest == null) {
                throw new IllegalStateException("Repository request is required");
            }

            return new ExportRequest(this);
        }

        @Nonnull
        public Builder exportLocation(String value) {
            exportLocation = value;
            return this;
        }

        @Nonnull
        public Builder repositoriesRequest(@Nonnull RepositoriesExportRequest value) {
            repositoriesRequest = requireNonNull(value, "repositoriesRequest");
            return this;
        }
    }
}
