package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.property.PropertySupport;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.Person;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Date;

/**
 * Describes a single commit. A commit represents a snapshot of a repository at a given point in time, with additional
 * metadata detailing who authored the commit and when, any message provided, and some number of parents (which may be
 * zero). Parents represent a commit's direct ancestors. Commits generally have a single parent, but:
 * <ul>
 *     <li>Commits which merge multiple streams of work may have multiple parents
 *     <ul>
 *         <li>For SCMs which support merge commits, the exact number of possible parents is SCM-specific</li>
 *     </ul>
 *     </li>
 *     <li>A repository will generally include a "root commit", the first commit to the repository, which will
 *     not have a parent
 *     <ul>
 *         <li>Whether or not a repository can have multiple root commits is SCM-specific</li>
 *     </ul>
 *     </li>
 * </ul>
 */
public interface Commit extends PropertySupport, MinimalCommit {

    /**
     * @return the person who authored the changes
     */
    @Nonnull
    Person getAuthor();

    /**
     * @return the timestamp when the changes were authored
     */
    @Nonnull
    Date getAuthorTimestamp();

    /**
     * @return the person who committed the changes
     * @since 5.0
     */
    @Nonnull
    default Person getCommitter() {
        return getAuthor();
    }

    /**
     * @return the timestamp when the changes were committed
     * @since 5.0
     */
    @Nonnull
    default Date getCommitterTimestamp() {
        return getAuthorTimestamp();
    }

    /**
     * @return the commit message
     */
    @Nullable
    String getMessage();

    /**
     * @return the commit's parents, which may be empty for the initial commit to a repository but will never be
     *         {@code null}
     */
    @Nonnull
    Collection<MinimalCommit> getParents();

    /**
     * Retrieves the {@link Repository repository} this commit was loaded from, if it's available.
     * <p>
     * <b>Note</b>: For reasons of performance and conciseness this field may be {@code null} depending on the service
     * used to construct this commit. If an explicit repository context has been provided by some other mechanism (for
     * example, web-fragments on the commit page will have an explicit 'repository' object in their render context) it
     * is preferable to use that instead of this.
     *
     * @return the repository that this commit belongs to
     */
    @Nullable
    Repository getRepository();
}
