package com.atlassian.bitbucket.watcher;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * A request object to search for {@link Watcher watcher} entities by {@link Watchable watchable}.
 * To check if a watcher exists for a given user / watchable pair see {@link IsWatchingRequest}
 *
 * @see WatcherService#search(WatcherSearchRequest, com.atlassian.bitbucket.util.PageRequest)
 * @since 5.10
 */
public class WatcherSearchRequest {

    private final boolean transitive;
    private final Watchable watchable;

    private WatcherSearchRequest(@Nonnull Builder builder) {
        transitive = builder.transitive;
        watchable = builder.watchable;
    }

    @Nonnull
    public Watchable getWatchable() {
        return watchable;
    }

    /**
     * Indicates if a transitive search should be performed.
     * <p>
     * Transitive searches will retrieve watchers within a hierarchy. Users who are watching multiple entities in a
     * hierarchy will be de-duplicated and only the most specific entity in the hierarchy will be returned.
     * For example, if a user is watching a {@link com.atlassian.bitbucket.repository.Repository repository} and a
     * {@link com.atlassian.bitbucket.pull.PullRequest pull request} only the pull request watcher will be returned.
     *
     * @return {@code true} if a transitive search should be performed. {@code false} otherwise
     */
    public boolean isTransitive() {
        return transitive;
    }

    public static class Builder {

        private final Watchable watchable;

        private boolean transitive;

        public Builder(@Nonnull Watchable value) {
            watchable = requireNonNull(value, "watchable");
        }

        @Nonnull
        public WatcherSearchRequest build() {
            return new WatcherSearchRequest(this);
        }

        @Nonnull
        public Builder transitive(boolean value) {
            transitive = value;

            return this;
        }
    }
}
