package com.atlassian.bitbucket.util;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * An instance of {@link Page} that supports pagination across an underlying paged collection.
 *
 * @param <T> type of the page
 */
public class FilteredPageImpl<T> implements Page<T> {

    private final boolean lastPage;
    private final int nextOrdinal;
    private final PageRequest pageRequest;
    private final int size;
    private final SortedMap<Integer, T> valueMap;

    public FilteredPageImpl(@Nonnull PageRequest pageRequest, int size, @Nonnull SortedMap<Integer, T> valueMap, boolean lastPage) {
        this.pageRequest = pageRequest;
        this.size = size;
        this.lastPage = lastPage;
        
        if (valueMap.size() > size) {
            nextOrdinal = Iterables.get(valueMap.keySet(), size);
            this.valueMap = valueMap.headMap(nextOrdinal);
        } else {
            this.valueMap = valueMap;
            if (valueMap.isEmpty()) {
                nextOrdinal = -1;
            } else {
                nextOrdinal = valueMap.lastKey() + 1;
            }
        }
    }

    /**
     * @param action the action to apply to each result
     * @since 6.3
     */
    @Override
    public void forEach(@Nonnull Consumer<? super T> action) {
        valueMap.values().forEach(action);
    }

    @Override
    public boolean getIsLastPage() {
        return lastPage;
    }

    @Override
    public int getLimit() {
        return pageRequest.getLimit();
    }

    @Override
    public PageRequest getNextPageRequest() {
        if (lastPage) {
            return null;
        } else {
            return new PageRequestImpl(nextOrdinal, pageRequest.getLimit());
        }
    }

    @Nonnull
    @Override
    public SortedMap<Integer, T> getOrdinalIndexedValues() {
        return valueMap;
    }

    @Override
    public int getSize() {
        return size;
    }

    @Override
    public int getStart() {
        return pageRequest.getStart();
    }

    @Nonnull
    @Override
    public Iterable<T> getValues() {
        return valueMap.values();
    }

    @Nonnull
    @Override
    public Stream<T> stream() {
        return valueMap.values().stream();
    }

    @Nonnull
    @Override
    public <E> FilteredPageImpl<E> transform(@Nonnull Function<? super T, ? extends E> transformFunction) {
        SortedMap<Integer, E> transformedValueMap = new TreeMap<>(Maps.transformValues(valueMap, transformFunction::apply));

        return new FilteredPageImpl<>(pageRequest, size, transformedValueMap, lastPage);
    }
}

