package com.atlassian.bitbucket.user;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.validation.annotation.RequiredString;

import javax.annotation.Nonnull;
import java.security.Principal;

/**
 * Represents a registered application user.
 */
public interface ApplicationUser extends Person, Principal {

    /**
     * The maximum length of a user slug
     */
    int MAX_SLUG_LENGTH = Project.MAX_KEY_LENGTH - 1;

    /**
     * The regex for user slugs
     */
    String SLUG_REGEXP = "[^\\\\/]+"; //anything besides a / or a \

    <T> T accept(@Nonnull ApplicationUserVisitor<T> visitor);

    /**
     * @return the user's display name
     */
    @Nonnull
    @RequiredString(size = 255)
    String getDisplayName();

    /**
     * @return the user id
     */
    int getId();

    /**
     * @return the type of user
     */
    @Nonnull
    UserType getType();

    /**
     * @return {@code true} if the user is enabled, {@code false} for deleted and disabled users.
     */
    boolean isActive();

    /**
     * Retrieves the "slug" for this user, which is a URL-friendly variant of its {@link #getName() name}. Each
     * user's slug is guaranteed to be unique within the system.
     * <p>
     * Although URL-friendly, appropriate encoding should still be performed on this slug whenever it is used in a URL.
     *
     * @return the user's slug
     */
    @Nonnull
    @RequiredString(size = MAX_SLUG_LENGTH, regexp = SLUG_REGEXP)
    String getSlug();
}
