package com.atlassian.bitbucket.topic;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Optional;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Settings for {@link TopicService#getTopic(String, TopicSettings) creating} a Topic. Allows the topic creator to
 * control the message queue size and the message deduplication policy.
 *
 * @since 5.3
 */
public class TopicSettings<T extends Serializable> {

    public static final int QUEUE_SIZE_DEFAULT = -1;

    private final boolean dedupePendingMessages;
    private final Class<T> messageType;
    private final int queueSize;

    private TopicSettings(Builder<T> builder) {
        dedupePendingMessages = builder.dedupePendingMessages;
        messageType = builder.messageType;
        queueSize = builder.queueSize;
    }

    @Nonnull
    public static <T extends Serializable> Builder<T> builder(@Nullable Class<T> messageType) {
        return new Builder<>(messageType);
    }

    /**
     * @return the type of {@link MessageEvent#getMessage() message} that will be sent to the topic, or
     *         {@link Optional#empty()} if none has been provided
     */
    @Nonnull
    public Optional<Class<T>> getMessageType() {
        return ofNullable(messageType);
    }

    /**
     * If the rate at which messages are published on the topic (temporarily) exceeds the rate at which the messages
     * are consumed, pending messages will be stored on the topic's message queue. This setting controls how many
     * messages can be stored on the queue. When the queue is full, messages will be dropped (and errors logged).
     *
     * @return the maximum number of messages that can be queued for the topic. A value of {@link #QUEUE_SIZE_DEFAULT}
     *         indicates that the default queue size will be used.
     */
    public int getQueueSize() {
        return queueSize;
    }

    /**
     * Whether identical {@link MessageEvent#getMessage() messages} <em>that haven't been dispatched yet</em> should be
     * deduplicated. Enabling this prevents the message queue from filling up with identical messages. The deduplication
     * only applies to pending messages and does not take into account any messages that have been dispatched to the
     * {@link TopicListener} in the past. Enabling this option is not appropriate for {@link TopicListener listeners}
     * that care about {@link MessageEvent#getPublishTime() timestamp} on the message, since deduplication only retains
     * the oldest (identical) message in the queue. Deduplication is performed by using the
     * {@link Object#equals(Object) equals} method on the {@link MessageEvent#getMessage() message}.
     *
     * @return whether messages should deduplicated in the message queue
     */
    public boolean isDedupePendingMessages() {
        return dedupePendingMessages;
    }

    public static class Builder<T extends Serializable> {

        private boolean dedupePendingMessages;
        private Class<T> messageType;
        private int queueSize;

        public Builder(@Nonnull TopicSettings<T> settings) {
            dedupePendingMessages = requireNonNull(settings, "settings").isDedupePendingMessages();
            messageType = settings.getMessageType().orElse(null);
            queueSize = settings.getQueueSize();
        }

        public Builder(@Nullable Class<T> messageType) {
            this.messageType = messageType;

            dedupePendingMessages = false;
            queueSize = QUEUE_SIZE_DEFAULT;
        }

        @Nonnull
        public TopicSettings<T> build() {
            return new TopicSettings<>(this);
        }

        @Nonnull
        public Builder<T> dedupePendingMessages(boolean value) {
            dedupePendingMessages = value;
            return this;
        }

        @Nonnull
        public Builder<T> queueSize(int value) {
            checkArgument(value == QUEUE_SIZE_DEFAULT || value > 0, "Queue size must be greater than 0");
            queueSize = value;
            return this;
        }
    }
}
