package com.atlassian.bitbucket.server;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;

/**
 * Core {@link Feature features} provided by the application
 *
 * @since 4.2
 */
public enum StandardFeature implements Feature {

    /**
     * Controls whether users can upload attachments in pull request comments. Disabling this feature will
     * <i>not</i> delete any attachments that have already been uploaded, but will prevent uploading more.
     */
    ATTACHMENTS("attachments"),
    /**
     * Controls whether users will be required to solve a CAPTCHA after a configured number of failed attempts
     * to login. Disabling this feature can reduce the overall security of an instance and is not recommended,
     * but may be necessary in some environments. If this feature is disabled, it is strongly advised that some
     * other aspect of the authentication mechanism (such as a remote LDAP directory) impose a limit on failed
     * attempts in order to prevent attackers from using brute force attacks to compromise accounts.
     */
    AUTH_CAPTCHA("auth.captcha"),
    /**
     * Controls whether Unicode bidirectional characters are highlighted in code contexts (source view, pull requests,
     * code blocks in comments, etc.). If enabled, these characters will be expanded (e.g. {@code <U+2066>}) so they
     * can be easily seen by reviewers.
     */
    BIDI_CHARACTER_HIGHLIGHTING("bidi.character.highlighting"),
    /**
     * Controls whether the commit graph is displayed on the commits page of a repository.
     *
     * @since 6.1
     */
    COMMIT_GRAPH("commit.graph"),
    /**
     * Controls whether diagnostics is enabled
     */
    DIAGNOSTICS("diagnostics"),
    /**
     * Controls whether admins have access to the Deployments view in the UI or the deployments endpoint
     * in REST. This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 7.16
     */
    DEPLOYMENTS("deployments", true),
    /**
     * Controls whether users can edit repository files in the browser and via REST. Disabling this feature
     * <i>does not</i> prevent plugins from using the Java API to edit files; it only disables editing via
     * the UI and REST.
     *
     * @see com.atlassian.bitbucket.content.ContentService#editFile
     * @since 4.13
     */
    FILE_EDITOR("file.editor"),
    /**
     * Controls whether repositories are allowed to be forked. When this feature is disabled, it overrides any
     * configuration applied to individual repositories.
     *
     * @see com.atlassian.bitbucket.repository.ForkingDisabledException
     * @see com.atlassian.bitbucket.repository.RepositoryService#fork
     */
    FORKS("forks"),
    /**
     * Controls whether new users will be presented with a "Getting Started" page after their first login.
     */
    GETTING_STARTED("getting.started.page"),
    /**
     * Controls whether Data Center Migration archives can be generated on this instance.
     * <p>
     * This is <i>not</i> a {@link #isDataCenter() data center-only} feature, data can be exported from Server
     * installations as well.
     *
     * @since 5.13
     */
    DATA_CENTER_MIGRATION_EXPORT("data.center.migration.export"),
    /**
     * Controls whether Data Center Migration archives can be imported into the instance.
     * <p>
     * This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 5.13
     */
    DATA_CENTER_MIGRATION_IMPORT("data.center.migration.import", true),
    /**
     * Controls whether the system can send development information to Jira Cloud.
     *
     * @since 7.14
     */
    JIRA_CLOUD_DEV_INFO("jira.cloud.devinfo", false),
    /**
     * Controls whether the Jira commit checker feature is enabled.
     * <p>
     * This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 7.16
     */
    JIRA_COMMIT_CHECKER("jira.commit.checker", true),
    /**
     * Controls whether users can create repositories in their personal projects. Disabling this feature will
     * prevent creating new repositories in, or forking existing repositories to, users' personal projects.
     * To completely disable forking, regardless of the target project's type, the {@link #FORKS} feature should
     * be disabled instead of, or in addition to, this one.
     *
     * @see com.atlassian.bitbucket.repository.PersonalRepositoryDisabledException
     */
    PERSONAL_REPOS("personal.repos"),
    /**
     * Controls whether HTTP access tokens at project and repository level are enabled.
     * <p>
     * This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 7.18
     */
    PROJECT_REPO_ACCESS_TOKENS("project.repo.access.tokens", true),
    /**
     * Controls whether {@link Project#isPublic() projects} and {@link Repository#isPublic() repositories} can be
     * configured to allow public access. When this feature is disabled, it overrides any configuration applied to
     * individual projects or repositories.
     */
    PUBLIC_ACCESS("public.access"),
    /**
     * Controls whether the process of automatically declining inactive pull requests is enabled for the system.
     * <p>
     * When this feature is enabled, all pull requests that are inactive (no recent comments, pushes etc.) are declined.
     * Individual projects or repositories are still able to opt-out or configure a different inactivity period.
     * When this feature is disabled, it overrides any configuration applied to individual projects or repositories,
     * and prevents any pull request from being automatically declining due to inactivity.
     *
     * @since 7.7
     */
    PULL_REQUEST_AUTO_DECLINE("pull.request.auto.decline"),
    /**
     * Controls whether pull requests can be deleted. When this feature is disabled, pull requests cannot be
     * disabled via plugins using the Java API or using the UI or REST.
     *
     * @see com.atlassian.bitbucket.pull.PullRequestDeletionDisabledException
     * @see com.atlassian.bitbucket.pull.PullRequestService#delete
     * @since 5.1
     */
    PULL_REQUEST_DELETION("pull.request.deletion"),
    /**
     * Controls whether pull request suggestions are enabled.
     * Enabling this feature allows users with write access to the source repository the ability to apply code changes
     * through the UI or the suggestion syntax. If this is disabled the user will not be able to apply code changes to
     * the source repository through the UI or using the suggestion syntax; it will default to a standard comment if the
     * suggestion syntax is used.
     *
     * @since 6.6
     */
    PULL_REQUEST_SUGGESTIONS("pull.request.suggestions"),
    /**
     * Controls whether or not the pull request templates feature is enabled.
     * Enabling this allows users to define a template for pull requests. If this is disabled, then users will be unable
     * to define pull request templates at any scope, via the UI or REST API. In addition, plugins will no longer be
     * able to interact with pull request templates.
     *
     * @since 7.13
     */
    PULL_REQUEST_TEMPLATES("pull.request.templates", true),
    /**
     * Controls whether HTTP requests will be rate limited per user.
     * If this is enabled, repeated HTTP requests from the same user in a short time period may be rate limited.
     * If this is disabled, no requests will be rate limited.
     *
     * @since 6.5
     */
    RATE_LIMITING("rate.limiting", true),
    /**
     * Controls whether the repository archiving feature is enabled. Archived repositories can no longer be pushed to,
     * and their settings can't be updated.
     * Archiving repositories is a {@link #isDataCenter() data center-only} feature, but archived repositories can be
     * unarchived even on a server license.
     *
     * @since 8.0
     */
    REPOSITORY_ARCHIVING("repository.archiving", true),
    /**
     * Controls whether a user can delete a repository by checking their permission level against the repository
     * delete policy.
     *
     * @since 7.4
     */
    REPOSITORY_DELETE_POLICY("repository.delete.policy", true),
    /**
     * Controls whether admins have access to the Repositories view in the UI or the repository-management endpoint
     * in REST. This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 7.12
     */
    REPOSITORY_MANAGEMENT("repository.management", true),
    /**
     * Controls whether admins have access to the Required Builds view in the UI or the required-builds endpoint
     * in REST. This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 7.14
     */
    REQUIRED_BUILDS("required.builds", true),
    /**
     * Controls whether a user can manage reviewer groups. This is a {@link #isDataCenter()
     * data center-only} feature.
     *
     * @since 7.13
     */
    REVIEWER_GROUPS("reviewer.groups", true),
    /**
     * Controls whether rolling upgrade can be performed for bugfix versions. This is a {@link #isDataCenter()
     * data center-only} feature.
     *
     * @since 7.9
     */
    ROLLING_UPGRADE("rolling.upgrade", true),
    /**
     * Controls whether secret scanning is enabled or not. This is a {@link #isDataCenter() data center-only} feature.
     *
     * @since 8.3
     */
    SECRET_SCANNING("secret.scanning", true),
    /**
     * Controls whether smart mirrors can be configured. This is a {@link #isDataCenter() data center-only} feature.
     * Enabling or disabling this feature has no effect if a data center license is not installed. Disabling this
     * feature prevents the use of smart mirrors even if a data center license <i>is</i> installed.
     */
    SMART_MIRRORS("smart.mirrors", true),
    /**
     * Controls enforce project settings selector on project admin settings
     *
     * @since 8.8
     */
    ENFORCE_PROJECT_SETTINGS("enforce.project.settings", true),
    /**
     * Controls whether users will be prompted to update their timezone after their first login.
     */
    TIME_ZONE_ONBOARDING("user.time.zone.onboarding");

    private final boolean dataCenter;
    private final String key;

    StandardFeature(String key) {
        this(key, false);
    }

    StandardFeature(String key, boolean dataCenter) {
        this.dataCenter = dataCenter;
        this.key = key;
    }

    @Nonnull
    public static StandardFeature fromKey(String value) {
        for (StandardFeature feature : values()) {
            if (feature.getKey().equals(value)) {
                return feature;
            }
        }
        throw new IllegalArgumentException("No Feature is associated with key [" + value + "]");
    }

    @Nonnull
    @Override
    public String getKey() {
        return key;
    }

    @Override
    public boolean isDataCenter() {
        return dataCenter;
    }
}
