package com.atlassian.bitbucket.scm.bulk;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.Set;

/**
 * Describes a callback for receiving a stream of {@link Commit commits}, each with a set listing one or more
 * {@link Repository repositories} where it was found.
 *
 * @since 5.8
 */
@FunctionalInterface
public interface BulkCommitCallback {

    /**
     * Receives a commit, and the set of repositories the commit was found in.
     * <p>
     * Because the {@link Commit} may be found in multiple repositories, {@link Commit#getRepository()} will always
     * return {@code null} on the provided commit. To know which repository, or repositories, the commit was actually
     * present in, use the accompanying set.
     * <p>
     * This method will never be invoked before {@link #onStart(BulkCommitContext)}, and if it is invoked
     * {@link #onEnd(BulkCommitSummary)} is guaranteed to be invoked, whether the end happens because this
     * method returns {@code false} or because the upstream sender runs out of commits.
     *
     * @param commit       the commit
     * @param repositories a set containing one or more repositories where the commit was found
     * @return {@code true} if more commits should be provided; otherwise {@code false} to end streaming,
     *        potentially leaving some number of commits unread
     * @throws IOException for implementations which perform I/O
     */
    boolean onCommit(@Nonnull Commit commit, @Nonnull Set<Repository> repositories) throws IOException;

    /**
     * Called after the final {@link #onCommit(Commit, Set) commit} has been streamed.
     * <p>
     * Note: If there were no commits, this method may be called immediately after {@link #onStart(BulkCommitContext)}
     * without any calls to {@link #onCommit(Commit, Set)}.
     *
     * @param summary summarizes the request and the streamed commits
     * @throws IOException for implementations which perform I/O
     */
    default void onEnd(@Nonnull BulkCommitSummary summary) throws IOException {
    }

    /**
     * Called before the first {@link #onCommit(Commit, Set) commit} is streamed.
     *
     * @param context provides details about the request for which commits are being streamed
     * @throws IOException for implementations which perform I/O
     */
    default void onStart(@Nonnull BulkCommitContext context) throws IOException {
    }
}
