package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.io.File;
import java.nio.file.Path;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * @since 7.11
 */
public class PushCommandParameters {

    private final boolean anonymous;
    private final Map<String, String> environment;
    private final boolean force;
    private final boolean includePrivateRefs;
    private final String password;
    private final File privateKeyFile;
    private final boolean prune;
    private final Set<String> refspecs;
    private final String remoteUrl;
    private final String username;

    private PushCommandParameters(PushCommandParameters.Builder builder) {
        environment = builder.environment.build();
        force = builder.force;
        includePrivateRefs = builder.includePrivateRefs;
        password = builder.password;
        privateKeyFile = builder.privateKeyFile;
        prune = builder.prune;
        remoteUrl = builder.remoteUrl;
        refspecs = builder.refspecs.build();
        username = builder.username;

        boolean credentialsSupplied = username != null || password != null || privateKeyFile != null;
        anonymous = builder.anonymous == null ? !credentialsSupplied : builder.anonymous;
        if (!anonymous && !credentialsSupplied) {
            throw new IllegalArgumentException("No credentials supplied, and 'anonymous' is not set");
        }
        if (anonymous && credentialsSupplied) {
            throw new IllegalArgumentException("Credentials supplied even though 'anonymous' is set");
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PushCommandParameters that = (PushCommandParameters) o;
        return anonymous == that.anonymous &&
               includePrivateRefs == that.includePrivateRefs &&
               force == that.force &&
               prune == that.prune &&
               Objects.equals(password, that.password) &&
               Objects.equals(privateKeyFile, that.privateKeyFile) &&
               Objects.equals(refspecs, that.refspecs) &&
               Objects.equals(remoteUrl, that.remoteUrl) &&
               Objects.equals(username, that.username);
    }

    /**
     * @return a map of environment variables that will be applied to the push command
     */
    @Nonnull
    public Map<String, String> getEnvironment() {
        return environment;
    }

    /**
     * @return the password to provide for authentication
     */
    @Nonnull
    public Optional<String> getPassword() {
        return ofNullable(password);
    }

    /**
     * @return the SSH private key to use for authentication
     */
    @Nonnull
    public Optional<File> getPrivateKey() {
        return ofNullable(privateKeyFile);
    }

    /**
     * @return a set of refspecs to be passed to the push command, returns an empty set if not set
     */
    @Nonnull
    public Set<String> getRefspecs() {
        return refspecs;
    }

    /**
     * @return the URL to push to
     */
    @Nonnull
    public String getRemoteUrl() {
        return remoteUrl;
    }

    /**
     * @return the username to provide for authentication
     */
    @Nonnull
    public Optional<String> getUsername() {
        return ofNullable(username);
    }

    @Override
    public int hashCode() {
        return Objects.hash(anonymous, includePrivateRefs, password, privateKeyFile, force, prune, refspecs, remoteUrl,
                username);
    }

    /**
     * @return whether the push command can be done in an anonymous context or not. If not provided, defaults to {@code false}
     */
    public boolean isAnonymous() {
        return anonymous;
    }

    /**
     * @return {@code true} when remote refs should be forced to update, {@code false} otherwise. If not provided,
     *         defaults to {@code false}
     */
    public boolean isForce() {
        return force;
    }

    /**
     * If an explicit refspec is set then this setting is ignored.
     * @return {@code true} if private refs (outside refs/heads and refs/tags) should be included in the push
     */
    public boolean isIncludePrivateRefs() {
        return includePrivateRefs;
    }

    /**
     * @return {@code true} when remote branches and tags should be deleted where the local branch or tag has been deleted.
     *         If not provided, defaults to {@code false}
     */
    public boolean isPrune() {
        return prune;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("includePrivateRefs", includePrivateRefs)
                .add("password", "<hidden>")
                .add("force", force)
                .add("prune", prune)
                .add("refspecs", refspecs)
                .add("remoteUrl", remoteUrl)
                .add("username", username)
                .toString();
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableMap.Builder<String, String> environment;
        private final ImmutableSet.Builder<String> refspecs;
        private final String remoteUrl;
        private Boolean anonymous;
        private boolean force;
        private boolean includePrivateRefs;
        private String password;
        private File privateKeyFile;
        private boolean prune;
        private String username;

        /**
         * @param remoteUrl the URL to push to
         */
        public Builder(@Nonnull String remoteUrl) {
            this.remoteUrl = requireNonBlank(remoteUrl, "remoteUrl");
            environment = ImmutableMap.builder();
            force = false;
            includePrivateRefs = true;
            prune = false;
            refspecs = ImmutableSet.builder();
        }

        /**
         * @param pushCommandParameters to copy
         */
        public Builder(@Nonnull PushCommandParameters pushCommandParameters) {
            //Don't copy the anonymous flag because credentials might still be set on the builder and we want
            //anonymous to default to the correct value.
            environment = ImmutableMap.<String, String>builder().putAll(pushCommandParameters.environment);
            remoteUrl = pushCommandParameters.getRemoteUrl();
            includePrivateRefs = pushCommandParameters.isIncludePrivateRefs();
            password = pushCommandParameters.getPassword().orElse(null);
            privateKeyFile = pushCommandParameters.getPrivateKey().orElse(null);
            prune = pushCommandParameters.isPrune();
            username = pushCommandParameters.getUsername().orElse(null);

            refspecs = ImmutableSet.builder();
            refspecs.addAll(pushCommandParameters.getRefspecs());
        }

        /**
         * If no credentials are to be provided, set this to {@code true}. If no credentials are provided, and anonymous
         * is not set to {@code true}, then an {@link IllegalArgumentException} will be thrown.
         * Similarly, if credentials are provided, but anonymous is set to {@code true} then an
         * {@link IllegalArgumentException} will be thrown.
         *
         * @param value whether or not the push command should provide credentials
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder anonymous(boolean value) {
            anonymous = value;
            return this;
        }

        @Nonnull
        public PushCommandParameters build() {
            return new PushCommandParameters(this);
        }

        /**
         * Specifies whether to supply {@code --force} to force updates to remote refs. Defaults to {@code false}.
         * @param value whether or not the push command should be forced
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder force(boolean value) {
            force = value;

            return this;
        }

        /**
         * If an explicit refspec is set then this setting is ignored.
         *
         * @param value whether to include private refs (outside refs/heads and refs/tags) in the push. The default
         *              value is true so all refs will be pushed.
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder includePrivateRefs(boolean value) {
            includePrivateRefs = value;
            return this;
        }

        /**
         * @param value the password to use
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder password(@Nonnull String value) {
            password = requireNonNull(value, "password");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder privateKey(@Nonnull File value) {
            privateKeyFile = requireNonNull(value, "privateKeyFile");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder privateKey(@Nonnull Path value) {
            return privateKey(requireNonNull(value, "privateKeyFile").toFile());
        }

        /**
         * Specifies whether to supply {@code --prune} to prune remote branches and tags where the local branch
         * or tag has been deleted.
         *
         * @param value true if pruning should be enabled
         * @return the builder
         */
        @Nonnull
        public PushCommandParameters.Builder prune(boolean value) {
            prune = value;
            return this;
        }

        /**
         * @param values refspecs to be passed to the push command
         * @return the builder
         */
        @Nonnull
        public PushCommandParameters.Builder refspecs(@Nonnull Iterable<String> values) {
            addIf(Objects::nonNull, refspecs, values);
            return this;
        }

        /**
         * @param value the username to use
         * @return {@code this}
         */
        @Nonnull
        public PushCommandParameters.Builder username(@Nonnull String value) {
            username = requireNonNull(value, "value");
            return this;
        }

        /**
         * Puts the provided {@code value} in the {@link #environment} map with the specified {@code name}, after ensuring
         * both the {@code name} and {@code value} are not blank.
         *
         * @param name  the name of the environment variable to set
         * @param value the value to set for the environment variable
         * @return the builder
         */
        @Nonnull
        public PushCommandParameters.Builder withEnvironment(@Nonnull String name, @Nonnull String value) {
            environment.put(requireNonBlank(name, "name"), requireNonBlank(value, "value"));
            return this;
        }
    }
}
