package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.*;
import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.Collections;
import java.util.Set;

import static com.atlassian.bitbucket.commit.AbstractCommitsRequest.UNLIMITED_MESSAGE_LENGTH;
import static java.util.Objects.requireNonNull;

public class CommitsCommandParameters extends AbstractCommandParameters {

    private final boolean all;
    private final Set<String> excludes;
    private final boolean followRenames;
    private final boolean ignoringMissing;
    private final Set<String> includes;
    private final int maxMessageLength;
    private final CommitListMergeFilter merges;
    private final CommitOrder order;
    private final Set<String> paths;
    private final Repository secondaryRepository;
    private final Instant since;
    private final boolean traversing;
    private final boolean withMessages;

    private CommitsCommandParameters(Builder builder) {
        all = builder.all;
        //If a traversal was not requested, ignore any excludes
        excludes = builder.traversing ? builder.excludes.build() : Collections.emptySet();
        followRenames = builder.followRenames;
        ignoringMissing = builder.ignoringMissing;
        //If "all" was specified, ignore any explicit includes
        includes = all ? Collections.emptySet() : builder.includes.build();
        //If builder.withMessages is true, normalize the requested length. Otherwise, ignore it. This means the
        //length can either be 0 because it was set that way explicitly, or because withMessages was false
        maxMessageLength = builder.withMessages ? Math.max(UNLIMITED_MESSAGE_LENGTH, builder.maxMessageLength) : 0;
        merges = builder.merges;
        order = builder.order;
        paths = builder.paths.build();
        secondaryRepository = builder.secondaryRepository;
        since = builder.since;
        traversing = builder.traversing;
        //Only set withMessages to true if builder.withMessages was true _and_ the max length isn't 0
        withMessages = builder.withMessages && maxMessageLength != 0;

        if (followRenames) {
            if (paths.isEmpty()) {
                throw new IllegalStateException("A path must be supplied when attempting to follow renames");
            }
            if (paths.size() > 1) {
                throw new IllegalStateException("Following renames is only possible with a single path");
            }
        }
    }

    @Nonnull
    public Set<String> getExcludes() {
        return excludes;
    }

    @Nonnull
    public Set<String> getIncludes() {
        return includes;
    }

    /**
     * @return the maximum length to read for {@link Commit#getMessage() commit messages}; if set, messages
     *         longer than this will be truncated; -1 indicates unlimited message length
     * @since 4.5
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    /**
     * Retrieves the strategy being used to filter merge commits.
     *
     * @return how merge commits will be filtered
     * @since 4.8
     * @see CommitListMergeFilter
     */
    @Nonnull
    public CommitListMergeFilter getMerges() {
        return merges;
    }

    /**
     * Retrieves the sort order to use for the commits.
     *
     * @return the sort order
     * @since 5.0
     */
    @Nonnull
    public CommitOrder getOrder() {
        return order;
    }

    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    /**
     * Show commits more recent than a specific date.
     *
     * @return the {@link Instant} which the commits should be more recent than or {@code null} if no filter is set
     * @since 7.12
     */
    @Nullable
    public Instant getSince() {
        return since;
    }

    public boolean hasExcludes() {
        return !excludes.isEmpty();
    }

    public boolean hasIncludes() {
        return !includes.isEmpty();
    }

    public boolean hasPaths() {
        return !paths.isEmpty();
    }

    public boolean hasSecondaryRepository() {
        return secondaryRepository != null;
    }

    /**
     * @return {@code true} if a since date is specified, {@code false} otherwise
     * @since 7.12
     */
    public boolean hasSince() {
        return since != null;
    }

    public boolean isAll() {
        return all;
    }

    /**
     * Whether or not the commit history will attempt to follow renames. This is only applicable for individual file
     * {@link #getPaths() paths}
     *
     * @return {@code true} if renames are followed, {@code false} otherwise
     *
     * @since 4.5
     */
    public boolean isFollowingRenames() {
        return followRenames;
    }

    /**
     * @return whether or not to ignore missing commits.
     *
     * @since 5.0
     */
    public boolean isIgnoringMissing() {
        return ignoringMissing;
    }

    public boolean isTraversing() {
        return traversing;
    }

    /**
     * @return {@code true} if commit messages should be included; otherwise, {@code false} if they should be omitted
     */
    public boolean isWithMessages() {
        return withMessages;
    }

    /**
     * Assembles a {@link CommitContext} from these parameters.
     *
     * @return a {@link CommitContext} describing these parameters
     */
    @Nonnull
    public CommitContext toContext() {
        return new CommitContext.Builder().build();
    }

    public static class Builder {

        private final ImmutableSet.Builder<String> excludes;
        private final ImmutableSet.Builder<String> includes;
        private final ImmutableSet.Builder<String> paths;

        private boolean all;
        private boolean followRenames;
        private boolean ignoringMissing;
        private int maxMessageLength;
        private CommitListMergeFilter merges;
        private CommitOrder order;
        private Repository secondaryRepository;
        private Instant since;
        private boolean traversing;
        private boolean withMessages;

        public Builder() {
            excludes = ImmutableSet.builder();
            includes = ImmutableSet.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
            merges = CommitListMergeFilter.INCLUDE;
            order = CommitOrder.DEFAULT;
            paths = ImmutableSet.builder();
            traversing = true;
            withMessages = true;
        }

        public Builder(@Nonnull CommitsBetweenRequest request) {
            this();

            exclude(requireNonNull(request, "request").getExcludes())
                    .followRenames(request.isFollowingRenames())
                    .ignoreMissing(request.isIgnoringMissing())
                    .include(request.getIncludes())
                    .maxMessageLength(request.getMaxMessageLength())
                    .merges(request.getMerges())
                    .order(request.getOrder())
                    .paths(request.getPaths())
                    .secondaryRepository(request.getSecondaryRepository());
        }

        public Builder(@Nonnull CommitsRequest request) {
            this();

            followRenames(requireNonNull(request, "request").isFollowingRenames())
                    .ignoreMissing(request.isIgnoringMissing())
                    .include(request.getCommitId())
                    .maxMessageLength(request.getMaxMessageLength())
                    .merges(request.getMerges())
                    .order(request.getOrder())
                    .paths(request.getPaths())
                    .since(request.getSince());
        }

        @Nonnull
        public Builder all(boolean value) {
            all = value;

            return this;
        }

        @Nonnull
        public CommitsCommandParameters build() {
            return new CommitsCommandParameters(this);
        }

        @Nonnull
        public Builder exclude(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, excludes, values);

            return this;
        }

        @Nonnull
        public Builder exclude(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, excludes, value, values);

            return this;
        }

        /**
         * @param value new value
         * @return {@code this}
         * @since 4.5
         */
        @Nonnull
        public Builder followRenames(boolean value) {
            followRenames = value;

            return this;
        }

        /**
         * @param value new value
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoringMissing = value;

            return this;
        }

        @Nonnull
        public Builder include(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, includes, values);

            return this;
        }

        @Nonnull
        public Builder include(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, includes, value, values);

            return this;
        }

        /**
         * @param value the maximum length for included commit messages, where {@code 0} omits messages entirely
         *              and {@code -1} includes the full message regardless of length
         * @return {@code this}
         * @since 4.5
         */
        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return this;
        }

        /**
         * @param value new value
         * @return {@code this}
         * @since 4.8
         */
        @Nonnull
        public Builder merges(@Nonnull CommitListMergeFilter value) {
            merges = requireNonNull(value, "merges");

            return this;
        }

        /**
         * @param value new value
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder order(@Nonnull CommitOrder value) {
            order = requireNonNull(value, "order");

            return this;
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return this;
        }

        @Nonnull
        public Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }

        /**
         * @param value limits to commits newer than the value provided
         * @return {@code this}
         * @since 7.12
         */
        @Nonnull
        public Builder since(@Nullable Instant value) {
            since = value;

            return this;
        }

        @Nonnull
        public Builder traverse(boolean value) {
            traversing = value;

            return this;
        }

        /**
         * @param value {@code true} to include commit messages in the output, up to the specified
         *              {@link #maxMessageLength(int) maximum length}; otherwise, {@code false} to
         *              omit messages entirely
         * @return {@code this}
         */
        @Nonnull
        public Builder withMessages(boolean value) {
            withMessages = value;

            return this;
        }
    }
}
