package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.NoSuchCommitException;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

public class ChangesetsCommandParameters extends AbstractCommandParameters {

    public static final int UNLIMITED_MESSAGE_LENGTH = -1;

    private final Set<String> commitIds;
    private final boolean ignoreMissing;
    private final int maxChangesPerCommit;
    private final int maxMessageLength;

    private ChangesetsCommandParameters(Builder builder) {
        commitIds = builder.commitIds.build();
        ignoreMissing = builder.ignoreMissing;
        maxChangesPerCommit = builder.maxChangesPerCommit;
        maxMessageLength = builder.maxMessageLength;

        if (commitIds.isEmpty()) {
            throw new IllegalStateException(
                    "At least one commit ID must be provided to retrieve changesets");
        }
        if (maxChangesPerCommit < 1) {
            throw new IllegalStateException(
                    "The maximum number of changes to include per commit must be greater than 0");
        }
    }

    @Nonnull
    public Set<String> getCommitIds() {
        return commitIds;
    }

    public int getMaxChangesPerCommit() {
        return maxChangesPerCommit;
    }

    /**
     * @return max length to process of a commit message. -1 for unlimited length
     * @since 4.5
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    /**
     * @return {@code true} if missing commits should be ignored; otherwise, {@code false} to throw a
     *         {@link NoSuchCommitException NoSuchCommitException} for missing commits
     */
    public boolean isIgnoreMissing() {
        return ignoreMissing;
    }

    public static class Builder {

        private final ImmutableSet.Builder<String> commitIds;

        private boolean ignoreMissing;
        private int maxChangesPerCommit;
        private int maxMessageLength;

        public Builder() {
            commitIds = ImmutableSet.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
        }

        @Nonnull
        public ChangesetsCommandParameters build() {
            return new ChangesetsCommandParameters(this);
        }

        @Nonnull
        public Builder commitId(@Nullable String value) {
            addIf(NOT_BLANK, commitIds, value);

            return this;
        }

        @Nonnull
        public Builder commitIds(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, commitIds, value, values);

            return this;
        }

        @Nonnull
        public Builder commitIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, commitIds, values);

            return this;
        }

        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoreMissing = value;

            return this;
        }

        @Nonnull
        public Builder maxChangesPerCommit(int value) {
            maxChangesPerCommit = value;

            return this;
        }

        @Nonnull
        /**
         * @since 4.5
         */
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;
            return this;
        }
    }
}
