package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.DiffContext;
import com.atlassian.bitbucket.content.DiffWhitespace;
import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Objects.requireNonNull;

@SuppressWarnings("unused") //Unused members are still public API
public class AbstractDiffCommandParameters extends AbstractCommandParameters {

    public static final int DEFAULT_CONTEXT_LINES = -1;

    //See BSERV-10916 to understand how this value was chosen
    private static final int MIN_LINE_LENGTH = 100;

    private final int contextLines;
    private final int maxLineLength;
    private final int maxLines;
    private final Set<String> paths;
    private final Repository secondaryRepository;
    private final DiffWhitespace whitespace;

    /**
     * @param builder a builder containing the assembled parameters
     * @since 5.4
     */
    protected AbstractDiffCommandParameters(AbstractBuilder<?> builder) {
        contextLines = builder.contextLines;
        maxLineLength = Math.max(builder.maxLineLength, MIN_LINE_LENGTH); //Enforce a min read length to avoid invalid values
        maxLines = builder.maxLines;
        paths = builder.paths.build();
        secondaryRepository = builder.secondaryRepository;
        whitespace = builder.whitespace;
    }

    public int getContextLines() {
        return contextLines;
    }

    public int getMaxLineLength() {
        return maxLineLength;
    }

    public int getMaxLines() {
        return maxLines;
    }

    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    /**
     * Retrieves the <b>secondary</b> repository from which to generate the diff. Commits in this repository may only
     * be identified <i>by hash</i>. Any branch or tag names used will <i>always</i> be resolved using the primary
     * repository.
     *
     * @return a secondary repository from which to generate the diff
     * @since 6.7
     */
    @Nullable
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    public boolean hasContextLines() {
        return contextLines != DEFAULT_CONTEXT_LINES;
    }

    public boolean hasPaths() {
        return !paths.isEmpty();
    }

    /**
     * @return optional flag to potentially ignore whitespace
     */
    @Nonnull
    public DiffWhitespace getWhitespace() {
        return whitespace;
    }

    /**
     * Appends common attributes to the provided {@link DiffContext.Builder} before building
     * it and returning the final {@link DiffContext}.
     *
     * @param builder the starting builder to apply common attributes to before building
     * @return the {@link DiffContext} from the provided builder
     */
    @Nonnull
    protected DiffContext toContext(@Nonnull DiffContext.Builder builder) {
        return requireNonNull(builder, "builder").contextLines(getContextLines())
                .maxLineLength(getMaxLineLength())
                .maxLines(getMaxLines())
                .whitespace(getWhitespace())
                .build();
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> {

        private final ImmutableSet.Builder<String> paths;

        private int contextLines;
        private int maxLineLength;
        private int maxLines;
        private Repository secondaryRepository;
        private DiffWhitespace whitespace;

        public AbstractBuilder() {
            contextLines = DEFAULT_CONTEXT_LINES;
            paths = ImmutableSet.builder();
            whitespace = DiffWhitespace.SHOW;
        }

        @Nonnull
        public B contextLines(int value) {
            contextLines = value;

            return self();
        }

        @Nonnull
        public B defaultContextLines() {
            contextLines = DEFAULT_CONTEXT_LINES;

            return self();
        }

        @Nonnull
        public B maxLineLength(int value) {
            maxLineLength = value;

            return self();
        }

        @Nonnull
        public B maxLines(int value) {
            maxLines = value;

            return self();
        }

        @Nonnull
        public B path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return self();
        }

        @Nonnull
        public B paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return self();
        }

        @Nonnull
        public B paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return self();
        }

        /**
         * @param value the <b>secondary</b> repository from which to generate the diff
         * @return {@code this}
         * @since 6.7
         */
        @Nonnull
        public B secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return self();
        }

        /**
         * @param value whitespace mode for showing diff
         * @return {@code this}
         */
        @Nonnull
        public B whitespace(@Nonnull DiffWhitespace value) {
            whitespace = requireNonNull(value, "whitespace");

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
