package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Describes a ref to {@link RefService#resolveRef(ResolveRefRequest) resolve}.
 * <p>
 * If it is known whether a requested ref is a branch or a tag, the caller is <i>encouraged</i> to specify the
 * {@link Builder#type type} to allow the SCM to optimize resolving the ref. If the ref type is unknown, the
 * type should be left {@code null}.
 *
 * @see StandardRefType
 * @since 4.6
 */
public class ResolveRefRequest {

    private final String refId;
    private final Repository repository;
    private final RefType type;

    private ResolveRefRequest(Builder builder) {
        refId = builder.refId;
        repository = builder.repository;
        type = builder.type;
    }

    /**
     * Retrieves the ID to resolve, which may be {@code empty()} to resolve the default branch. If the ID is
     * {@code empty()}, any {@link #getType type} specified is <i>ignored</i>.
     *
     * @return the ID to resolve, or {@code empty()} to resolve the repository's default branch
     */
    @Nonnull
    public Optional<String> getRefId() {
        return ofNullable(refId);
    }

    /**
     * Retrieves the repository to resolve the ref in.
     *
     * @return the repository to resolve the ref in
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * Retrieves the type of ref to resolve, if it's known in advance. Specifying the type explicitly can allow
     * the implementation to search IDs more efficiently, and can also prevent ambiguities when refs of multiple
     * types have the same ID.
     * <p>
     * If {@link #getRefId()} is {@code empty()}, this type hint is <i>ignored</i> and the default branch will
     * be resolved.
     *
     * @return the type of ref to resolve, if known in advance, or {@code empty()} to allow resolving
     *         a {@link Branch} or a {@link Tag}
     * @see StandardRefType
     */
    @Nonnull
    public Optional<RefType> getType() {
        return ofNullable(type);
    }

    public static class Builder extends BuilderSupport {

        private final Repository repository;

        private String refId;
        private RefType type;

        public Builder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");
        }

        @Nonnull
        public ResolveRefRequest build() {
            return new ResolveRefRequest(this);
        }

        /**
         * @param value the ref ID to resolve, or {@code null} to resolve the default branch
         * @return {@code this}
         */
        @Nonnull
        public Builder refId(@Nullable String value) {
            refId = value;

            return this;
        }

        /**
         * @param value the type of ref to resolve, or {@code null} if the type is not known in advance
         * @return {@code this}
         * @see StandardRefType
         */
        @Nonnull
        public Builder type(@Nullable RefType value) {
            type = value;

            return this;
        }
    }
}
