package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.repository.Repository.State;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Request for searching repositories. The purpose and behaviour of each field is described on its accessor.
 * <p>
 * The term "prefix-match", when used in this documentation, describes a string matching process that satisfies the
 * following criteria:
 * <ul>
 *     <li>the matched value <i>starts with</i> the provided filter value</li>
 *     <li>the matching is performed in a case-insensitive manner</li>
 *     <li>the filter string's leading and trailing whitespace characters are ignored</li>
 *     <li>a <i>blank</i> filter ({@literal null}, or whitespace characters only) is treated as 'empty' and results in
 *     <i>all</i> repositories being returned for the purpose of that particular filter (note that other filters
 *     specified in conjunction with that filter may still limit the final result)</li>
 * </ul>
 */
public class RepositorySearchRequest {

    private final RepositoryArchiveFilter archived;
    private final String name;
    private final Permission permission;
    private final String projectName;
    private final String projectKey;
    private final State state;
    private final RepositoryVisibility visibility;

    private RepositorySearchRequest(String name, String projectName, String projectKey, Permission permission,
                                    RepositoryVisibility visibility, State state, RepositoryArchiveFilter archived) {
        this.archived = archived;
        this.name = name;
        this.permission = permission;
        this.projectName = projectName;
        this.projectKey = projectKey;
        this.state = state;
        this.visibility = visibility;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those, whose {@code boolean} archive state
     * match the provided value.
     * <p>
     * The provided value may be {@code null}, in which case no additional restriction is applied to the search.
     *
     * @return the required boolean archive state, or {@code null}
     * @since 8.0
     */
    @Nullable
    public RepositoryArchiveFilter getArchived() {
        return archived;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those that <em>prefix-match</em> the provided
     * value according to the rules described in the class header.
     * <p>
     * If this field is not set, it is assumed that all repositories shall be returned, regardless of the repository
     * name. Applying the {@code projectName} filter would still limit the results to matching project names in such
     * case.
     *
     * @return filter text to match against repository names (and possibly project names), or {@code null} to return
     * repositories regardless of name.
     * @see #getProjectName()
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those, whose project's name <em>prefix-match</em>
     * the provided value.
     * <p>
     * If this field is not set, it is assumed that all repositories shall be returned, regardless of the project name.
     * Applying other filters would still limit the final results in such case.
     *
     * @return filter text to match against project names, or {@literal null} to return repositories regardless of
     * the project name
     * @see #getName()
     */
    @Nullable
    public String getProjectName() {
        return projectName;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those, whose project's key <em>exact-match</em>
     * (case-insensitive) the provided value.
     * <p>
     * If this field is not set, it is assumed that all repositories shall be returned, regardless of the project key.
     * Applying other filters would still limit the final results in such case.
     *
     * @return the required repository project key, or {@code null}
     * @since 8.0
     */
    @Nullable
    public String getProjectKey() {
        return projectKey;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those that are in the specified state. The
     * provided value may be {@code null}, in which case no additional restriction is applied to the search.
     *
     * @return the required repository state, or {@code null}
     * @since 5.13
     */
    @Nullable
    public State getState() {
        return state;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those for which the current user has the
     * required {@link Permission}. The provided value may be {@code null}, in which case a default permission
     * {@link Permission#REPO_READ} will be used.
     *
     * @return the required repository permission, or {@code null} to default to {@link Permission#REPO_READ}
     */
    @Nullable
    public Permission getPermission() {
        return permission;
    }

    /**
     * When set, limits returned {@link Repository repositories} to only those which match the specified
     * {@link RepositoryVisibility visibility}. The provided value may be {@code null}, in which case no
     * additional restriction is applied to the search.
     *
     * @return the required repository visibility, or {@code null} to return repositories regardless of visibility.
     * @see RepositoryVisibility
     */
    @Nullable
    public RepositoryVisibility getVisibility() {
        return visibility;
    }

    /**
     * Retrieves a flag indicating whether {@link #getArchived() archived} text has been set.
     *
     * @return {@code true} if {@link #getArchived()} is not {@code null}, {@code false} otherwise
     * @since 8.0
     */
    public boolean hasArchived() {
        return archived != null;
    }

    /**
     * Retrieves a flag indicating whether {@link #getName() name} text has been set.
     *
     * @return {@code true} if {@link #getName()} is not {@code null} or whitespace, {@code false} otherwise
     * @see #getName()
     */
    public boolean hasName() {
        return name != null;
    }

    /**
     * Retrieves a flag indicating whether {@link #getProjectName()} projectName} text has been set.
     *
     * @return {@code true} if {@link #getName()} is not {@code null} or whitespace, {@code false} otherwise
     */
    public boolean hasProjectName() {
        return projectName != null;
    }

    /**
     * Retrieves a flag indicating whether {@link #getProjectKey()} projectKey} text has been set.
     *
     * @return {@code true} if {@link #getProjectKey()} is not {@code null} or whitespace, {@code false} otherwise
     * @since 8.0
     */
    public boolean hasProjectKey() {
        return projectKey != null;
    }

    /**
     * Retrieves a flag indicating whether a specific {@link #getPermission() permission} has been set. If no explicit
     * permission has been set, the search will default to {@link Permission#REPO_READ}.
     *
     * @return {@code true} if {@link #getPermission()} is not {@code null}, {@code false} otherwise
     * @see #getPermission()
     */
    public boolean hasPermission() {
        return permission != null;
    }

    /**
     * Retrieves a flag indicating whether {@link #getState()} state} has been set.
     *
     * @return {@code true} if {@link #getState()} is not {@code null}, {@code false} otherwise
     * @since 5.13
     */
    public boolean hasState() {
        return state != null;
    }

    /**
     * Retrieves a flag indicating whether a specific {@link #getVisibility() visibility} has been set.
     *
     * @return {@code true} if {@link #getVisibility()} is not {@code null}, {@code false} otherwise
     * @see #getVisibility()
     */
    public boolean hasVisibility() {
        return visibility != null;
    }

    /**
     * Checks whether this request is considered empty, that is, none of the filters have been set.
     *
     * @return {@literal true}, if none of the filter fields on this {@code request} have been set, {@literal false}
     * otherwise
     * @see #hasName()
     * @see #hasProjectName()
     * @see #hasPermission()
     * @see #hasVisibility()
     */
    public boolean isEmpty() {
        return !hasName() && !hasProjectName() && !hasPermission() && !hasState() && !hasVisibility();
    }

    public static class Builder {

        private RepositoryArchiveFilter archived = RepositoryArchiveFilter.ACTIVE;
        private String name;
        private String projectName;
        private String projectKey;
        private Permission permission;
        private State state;
        private RepositoryVisibility visibility;

        public Builder() {
        }

        /**
         * Constructs a new {@code Builder} which will copy initial values from the provided
         * {@link RepositorySearchRequest request}.
         *
         * @param request request to copy
         */
        public Builder(@Nonnull RepositorySearchRequest request) {
            archived = request.getArchived();
            name = requireNonNull(request, "request").getName();
            permission = request.getPermission();
            projectKey = request.getProjectKey();
            state = request.getState();
            visibility = request.getVisibility();
        }

        /**
         * Assembles a new {@link RepositorySearchRequest} from the provided values.
         *
         * @return a new repository request instance
         */
        @Nonnull
        public RepositorySearchRequest build() {
            return new RepositorySearchRequest(name, projectName, projectKey, permission, visibility, state, archived);
        }

        /**
         * Set the repository archive state filter for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value filter archive state for the repository search; may be {@code null} or blank, which will both be
         *              treated as an empty filter
         * @return {@code this}
         * @see RepositorySearchRequest#getArchived()
         * @since 8.0
         */
        @Nonnull
        public Builder archived(@Nullable RepositoryArchiveFilter value) {
            archived = value;

            return this;
        }

        /**
         * Set the name filter for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value filter text for the repository search; may be {@code null} or blank, which will both be treated
         *              as an empty filter
         * @return {@code this}
         * @see RepositorySearchRequest#getName()
         */
        @Nonnull
        public Builder name(@Nullable String value) {
            name = StringUtils.trimToNull(value);

            return this;
        }

        /**
         * Set the project name filter for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value filter text for the repository project name; may be {@code null} or blank, which will both be
         *              treated as an empty filter
         * @return {@code this}
         * @see RepositorySearchRequest#getProjectName()
         */
        @Nonnull
        public Builder projectName(@Nullable String value) {
            projectName = StringUtils.trimToNull(value);

            return this;
        }

        /**
         * Set the project key filter for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value filter text for the repository project key; may be {@code null} or blank, which will both be
         *              treated as an empty filter
         * @return {@code this}
         * @see RepositorySearchRequest#getProjectKey()
         * @since 8.0
         */
        @Nonnull
        public Builder projectKey(@Nullable String value) {
            projectKey = StringUtils.trimToNull(value);

            return this;
        }

        /**
         * Set the permission level for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value the required permission for the repository search; may be {@code null}
         * @return {@code this}
         * @throws IllegalArgumentException if the provided {@link Permission} is not
         *                                  {@link Permission#isResource(Class) repository related}
         * @see RepositorySearchRequest#getPermission()
         */
        @Nonnull
        public Builder permission(@Nullable Permission value) {
            checkArgument(value == null || value.isResource(Repository.class),
                    "The provided permission is not valid for repositories");
            permission = value;

            return this;
        }

        /**
         * Set the state filter for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value the required state for the repository search; may be {@code null}
         * @return {@code this}
         * @see RepositorySearchRequest#getState()
         * @since 5.13
         */
        @Nonnull
        public Builder state(@Nullable State value) {
            state = value;

            return this;
        }

        /**
         * Set the visibility level for the resulting {@link RepositorySearchRequest request} instance.
         *
         * @param value the required visibility for the repository search; may be {@code null}
         * @return {@code this}
         * @see RepositorySearchRequest#getVisibility()
         */
        @Nonnull
        public Builder visibility(@Nullable RepositoryVisibility value) {
            visibility = value;

            return this;
        }
    }
}
