package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.hook.repository.RepositoryHookVetoedException;
import com.atlassian.bitbucket.scm.FeatureUnsupportedScmException;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * A service for interacting with a {@link Repository repository's} {@link Ref refs}, such as {@link Branch branches}
 * and {@link Tag tags}.
 */
public interface RefService {

    /**
     * Create branch with the information provided in the {@link CreateBranchRequest request}.
     *
     * @param request the branch creation request
     * @return the created branch
     * @throws RepositoryHookVetoedException if branch creation was canceled by one or more {@code repository-hooks}
     */
    @Nonnull
    Branch createBranch(@Nonnull CreateBranchRequest request);

    /**
     * Create tag with the information provided in the {@link CreateTagRequest request}.
     *
     * @param request the tag creation request
     * @return the created tag
     * @throws RepositoryHookVetoedException if tag creation was canceled by one or more {@code repository-hooks}
     */
    @Nonnull
    Tag createTag(@Nonnull CreateTagRequest request);

    /**
     * Retrieves a paged list of {@link Branch branches} for the specified repository, optionally filtered by the
     * provided starting text.
     *
     * @param request     request parameters for this query, including repository, filter text and ordering
     * @param pageRequest the page request defining the page start and limit
     * @return a page containing 0 or more branches
     */
    @Nonnull
    Page<Branch> getBranches(@Nonnull RepositoryBranchesRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves the default branch for the specified repository.
     * <p>
     * If the repository is newly created and no commits have been pushed to it, its default branch will not yet
     * exist. As a result, for new repositories, this method can throw {@link NoDefaultBranchException}. If you
     * want to know what the configured default branch is, regardless of whether the ref actually exists, use
     * {@link RepositoryService#getDefaultBranch(Repository)} instead.
     *
     * @param repository the repository to retrieve the default branch for
     * @return the default branch
     * @throws NoDefaultBranchException when no default branch is configured for the repository
     * @see RepositoryService#getDefaultBranch(Repository)
     */
    @Nonnull
    Branch getDefaultBranch(@Nonnull Repository repository) throws NoDefaultBranchException;

    /**
     * Retrieves the associated metadata based on a given collection of refs in a repository.
     *
     * @param request request parameters for this query, including repository and refs to find metadata for
     * @return a containing the associated metadata keyed by {@link Ref}
     */
    @Nonnull
    Map<Ref, MetadataMap> getMetadataByRefs(@Nonnull RefMetadataRequest request);

    /**
     * Retrieves a paged list of {@link Tag tags} for the specified repository, optionally filtered by the provided
     * starting text.
     *
     * @param request     request parameters for this query, including repository, filter text and ordering
     * @param pageRequest the page request defining the page start and limit
     * @return a page containing 0 or more tags
     */
    @Nonnull
    Page<Tag> getTags(@Nonnull RepositoryTagsRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Attempts to resolve a {@link Ref} from the provided {@link ResolveRefRequest#getRefId() refId}. Specifically,
     * the object is resolved as either a {@link Branch} or {@link Tag}. If the provided {@code refId} references
     * something else, such as a commit hash, tree or blob, {@code null} is returned.
     * <p>
     * If the ID is expected to match a ref of a specific {@link RefType type}, the request may be explicitly limited
     * to that {@link ResolveRefRequest#getType type}.
     * <p>
     * The exact behavior of this method may vary between SCMs. For example, when provided with a commit hash, the
     * SCM implementor may choose to return a {@link Branch} or {@link Tag} which references that commit, instead of
     * returning {@code null}.
     *
     * @param request describes the ID to resolve, and the repository to resolve it in
     * @return a {@link Branch} or a {@link Tag} or {@code null} to indicate "something else"
     * @since 4.6
     */
    @Nullable
    Ref resolveRef(@Nonnull ResolveRefRequest request);

    /**
     * Attempts to resolve multiple refs at once, returning a map with the <i>successfully resolved</i> IDs linked
     * to their resolved {@link Ref refs}.
     * <p>
     * If an ID is expected to match a {@link ResolveRefsRequest#getBranchIds() branch} or
     * {@link ResolveRefsRequest#getTagIds() tag}, it should be included in the relevant set. IDs for which the type
     * is not known in advance should be included in the {@link ResolveRefsRequest#getRefIds generic ref set}.
     * <p>
     * The exact behavior of this method may vary between SCMs. For example, when provided with a commit hash, the
     * SCM implementor may choose to resolve it to a {@link Branch} or {@link Tag} which references that commit, or
     * they may choose to always omit it from the returned map.
     *
     * @param request describes the branch, tag and unspecified ref IDs to resolve, and the repository to resolve
     *                them in
     * @return a map containing the <i>successfully resolved</i> IDs, which may be empty but never {@code null}
     * @since 4.6
     */
    @Nonnull
    Map<String, Ref> resolveRefs(@Nonnull ResolveRefsRequest request);

    /**
     * Sets the default branch for the specified repository. Support for this operation is SCM-specific, and different
     * SCMs may apply different rules to what values are supported.
     *
     * @param repository the repository to update the default branch for
     * @param branchName the branch to mark as the default within the repository
     * @throws FeatureUnsupportedScmException if the SCM for the specified repository does not
     *                                                                support updating the default branch
     */
    void setDefaultBranch(@Nonnull Repository repository, @Nonnull String branchName);

    /**
     * Streams {@link Branch branches} to a callback class, for the specified repository, and where the branches match
     * the request.
     *
     * @param request   request parameters for this query, including repository, filter text and ordering
     * @param callback  a callback to receive the {@link Branch branches}
     * @since 5.1
     */
    void streamBranches(@Nonnull RepositoryBranchesRequest request, @Nonnull BranchCallback callback);

    /**
     * Streams {@link Tag tags} to a callback class, for the specified repository, and where the tags match the request.
     *
     * @param request   request parameters for this query, including repository, filter text and ordering
     * @param callback  a callback to receive the {@link Tag tags}
     */
    void streamTags(@Nonnull RepositoryTagsRequest request, @Nonnull TagCallback callback);
}
