package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * A callback for streaming {@link Branch branches}.
 *
 * @since 4.6
 */
public interface BranchCallback {

    /**
     * Called to process a {@link Branch branch}.
     *
     * @param branch the {@link Branch branch} to stream
     * @return {@code true} if further branches should be streamed; otherwise, {@code false} to stop streaming
     * @throws IOException may be thrown by implementations which perform I/O.
     */
    default boolean onBranch(@Nonnull Branch branch) throws IOException {
        return true;
    }

    /**
     * Called after the final {@link #onBranch branch} has been streamed.
     * <p>
     * Note: If there were no refs, this method may be called immediately after {@link #onStart(BranchContext)}
     * without any calls to {@link #onBranch(Branch)}.
     *
     * @param summary summarizes the request and the streamed refs
     * @throws IOException may be thrown by implementations which perform I/O.
     */
    default void onEnd(@Nonnull BranchSummary summary) throws IOException {
    }

    /**
     * Called before the <i>first</i> {@link #onBranch branch} is streamed.
     *
     * @param context provides details about the request for which refs are being streamed
     * @throws IOException may be thrown by implementations which perform I/O.
     */
    default void onStart(@Nonnull BranchContext context) throws IOException {
    }
}
