package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.comment.Commentable;
import com.atlassian.bitbucket.property.PropertySupport;
import com.atlassian.bitbucket.validation.annotation.OptionalString;
import com.atlassian.bitbucket.validation.annotation.RequiredString;
import com.atlassian.bitbucket.watcher.Watchable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;
import java.util.Set;

public interface PullRequest extends PropertySupport, Commentable, Watchable {

    /**
     * @return the creating author of the pull request
     */
    @Nonnull
    PullRequestParticipant getAuthor();

    /**
     * @return the date the pull request was closed (i.e. last transitioned to {@link PullRequestState#MERGED} or
     * {@link PullRequestState#DECLINED}; or {@code null} if the pull request is currently {@link PullRequestState#OPEN}
     * @since 4.10
     */
    @Nullable
    Date getClosedDate();

    /**
     * @return the date when the pull request was created
     */
    @Nonnull
    Date getCreatedDate();

    /**
     * @return the description associated with the pull request (limit to 32k) or {@code null} if none was specified
     */
    @Nullable
    @OptionalString(size = 32768) //32k
    String getDescription();

    /**
     * @return the ref containing the changes to review and merge
     */
    @Nonnull
    PullRequestRef getFromRef();

    /**
     * @return the repository-scoped identifier of the pull request
     */
    long getId();

    /**
     * @return the non-reviewing participants attached to the pull request
     */
    @Nonnull
    Set<PullRequestParticipant> getParticipants();

    /**
     * @return the reviewing participants attached to the pull request
     */
    @Nonnull
    Set<PullRequestParticipant> getReviewers();

    /**
     * @return the state (e.g. opened, merged or declined) of the pull request
     */
    @Nonnull
    PullRequestState getState();

    /**
     * @return the mandatory title of the pull request
     */
    @Nonnull
    @RequiredString
    String getTitle();

    /**
     * @return the ref the changes should be merged to
     */
    @Nonnull
    PullRequestRef getToRef();

    /**
     * @return the last update of the pull request
     */
    @Nonnull
    Date getUpdatedDate();

    /**
     * @return the current version of the pull request
     */
    int getVersion();

    /**
     * Retrieves a flag indicating whether this pull request is closed. {@link PullRequestState#DECLINED Declined} and
     * {@link PullRequestState#MERGED merged} pull requests are both considered closed.
     *
     * @return {@code true} if this pull request is declined or merged, otherwise, {@code false} if it is
     *         {@link #isOpen() open}
     */
    boolean isClosed();

    /**
     * Retrieves a flag indicating whether the {@link #getFromRef() from} and {@link #getToRef() to} repositories for
     * this pull request are distinct. This is a convenience utility to allow simple detection for pull requests that
     * merge changes from one repository into another.
     *
     * @return {@code true} if this pull request will merge changes from one repository into another one; otherwise,
     *         {@code false} if the changes are simply on different branches in the same repository
     */
    boolean isCrossRepository();

    /**
     * @return {@code true} if the pull request is locked internally; otherwise {@code false}
     */
    boolean isLocked();

    /**
     * Retrieves a flag indicating whether this pull request is {@link PullRequestState#OPEN open}. Pull requests
     * which have been {@link PullRequestState#DECLINED declined} or {@link PullRequestState#MERGED merged} are
     * considered {@link #isClosed() closed}.
     *
     * @return {@code true} if this pull request is open; otherwise, {@code false} if is declined or merged
     */
    boolean isOpen();
}
