package com.atlassian.bitbucket.permission;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Used to set a permission to multiple users and/or groups.
 *
 * @see PermissionAdminService#setPermission(SetPermissionRequest)
 */
public class SetPermissionRequest {

    private final Permission permission;
    private final Set<ApplicationUser> users;
    private final Set<String> groups;
    private final Project project;
    private final Repository repository;

    private SetPermissionRequest(@Nonnull Permission permission, @Nonnull Set<ApplicationUser> users, @Nonnull Set<String> groups,
                                 @Nullable Project project, @Nullable Repository repository) {
        checkArgument(!(permission.isGlobal() && (project != null || repository != null)), "global permissions can not be granted to projects or repositories");
        checkArgument(!(permission.isResource(Project.class) && project == null), "project permissions can not be granted to a null project");
        checkArgument(!(permission.isResource(Repository.class) && repository == null), "repository permissions can not be granted to a null repository");
        checkArgument(!(permission.isResource(Project.class) && repository != null), "project permissions can not be granted to a repository");
        checkArgument(!(permission.isResource(Repository.class) && project != null), "repository permissions can not be granted to a project");
        checkArgument(!(users.isEmpty() && groups.isEmpty()), "either a user or a group must be specified");

        this.permission = permission;
        this.groups = groups;
        this.users = users;

        this.project = project;
        this.repository = repository;
    }

    @Nonnull
    public Permission getPermission() {
        return permission;
    }

    @Nonnull
    public Set<ApplicationUser> getUsers() {
        return users;
    }

    @Nonnull
    public Set<String> getGroups() {
        return groups;
    }

    public Project getProject() {
        return project;
    }

    public Repository getRepository() {
        return repository;
    }

    @Override
    public int hashCode() {
        return Objects.hash(permission, users, groups, project, repository);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || getClass() != obj.getClass()) {
            return false;
        }
        final SetPermissionRequest other = (SetPermissionRequest) obj;
        return Objects.equals(this.permission, other.permission) && Objects.equals(this.users, other.users) &&
                Objects.equals(this.groups, other.groups) && Objects.equals(this.project, other.project) &&
                Objects.equals(this.repository, other.repository);
    }

    public static class Builder extends BuilderSupport {

        private Permission permission;
        private Set<ApplicationUser> users;
        private Set<String> groups;
        private Project project;
        private Repository repository;

        public Builder() {
        }

        public Builder globalPermission(@Nonnull Permission permission) {
            checkArgument(permission.isGlobal(), "not a global permission");
            this.permission = permission;
            return this;
        }

        public Builder projectPermission(@Nonnull Permission permission, @Nonnull Project project) {
            checkArgument(permission.isResource(Project.class), "not a project permission");
            this.permission = permission;
            this.project = requireNonNull(project, "project");
            return this;
        }

        public Builder repositoryPermission(@Nonnull Permission permission, @Nonnull Repository repository) {
            checkArgument(permission.isResource(Repository.class), "not a repository permission");
            this.permission = permission;
            this.repository = requireNonNull(repository, "repository");
            return this;
        }

        public Builder user(@Nonnull ApplicationUser user) {
            // use a Precondition rather than relying on the NPE thrown by ImmutableSet.of(null)
            this.users = ImmutableSet.of(requireNonNull(user, "user"));
            return this;
        }

        public Builder users(@Nonnull Iterable<ApplicationUser> users) {
            // use a Precondition rather than relying on the NPE thrown by ImmutableSet.copyOf({null})
            checkArgument(Iterables.all(users, Objects::nonNull), "can not have null users");
            this.users = ImmutableSet.copyOf(users);
            return this;
        }

        public Builder group(@Nonnull String group) {
            checkNotBlank(group, "group");
            this.groups = ImmutableSet.of(group);
            return this;
        }

        public Builder groups(@Nonnull Iterable<String> groups) {
            checkArgument(Iterables.all(groups, StringUtils::isNotEmpty), "can not have null or empty groups");
            this.groups = ImmutableSet.copyOf(groups);
            return this;
        }

        public SetPermissionRequest build() {
            Set<ApplicationUser> users = this.users == null ? ImmutableSet.<ApplicationUser>of() : this.users;
            Set<String> groups = this.groups == null ? ImmutableSet.<String>of() : this.groups;
            return new SetPermissionRequest(permission, users, groups, project, repository);
        }
    }

}
