package com.atlassian.bitbucket.nav;

import com.atlassian.bitbucket.content.ArchiveFormat;
import com.atlassian.bitbucket.content.Path;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.repository.Ref;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositoryVisibility;
import com.atlassian.bitbucket.server.ApplicationPropertiesService;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.UserType;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Helps build URLs using a fluent interface. All returned URLs are URL encoded.
 *
 * Example usages:
 * <ul>
 * <li>nb.allProjects().startAt(0).withLimit(25).buildAbsolute() =&gt; http://localhost:7990/bitbucket/projects?start=0&amp;limit=25</li>
 * <li>nb.project("EX").repo("example").settings().buildRelative() =&gt; /stash/projects/EX/repos/example/settings</li>
 * <li>nb.project("EX").repo("example").browse().resource("foo/bar/baz.xml").buildNoContext() =&gt; /projects/EX/repos/example/browse/foo/bar/baz.xml</li>
 * <li>nb.project("EX").repo("example").browse().resource("foo/bar/baz.xml").onBranch("stash1.0").buildNoContext()
 *     =&gt; /projects/EX/repos/example/browse/foo/bar/baz.xml?branchName=stash1.0</li>
 * <li>nb.project("EX").repo("example").pullRequest("7636").buildNoContext() =&gt; /projects/EX/repos/example/pull-request/7636</li>
 * </ul>
 */
@SuppressWarnings("unused")
public interface NavBuilder {

    /**
     * A key which is wrapped in {@link #TOKEN_PREFIX} and {@link #TOKEN_SUFFIX} to produce a base URL token.
     * <p>
     * The key used needs to contain (preferably end with) something which is escaped when URL-encoded. Doing so
     * causes the end token to be different depending on whether the base URL has been encoded and allows callers
     * to encode the value they replace it with. That, in turn, ensures the resulting full URL has correct encoding
     * from beginning to end.
     * <p>
     * Over and above containing something which requires encoding, the value needs to encode <i>reliably</i>. That
     * means if the token has been encoded 4 times, any caller performing the substitution should be able to detect
     * that count, because they will need to encode the replacement value 4 times over as well.
     * <p>
     * The current choice is a single trailing %. On first encoding, this produces "%25"; second produces "%2525",
     * third produces "%252525", etc. This allows for two use cases:
     * <ol>
     *     <li><i>I don't care about encoding</i>: Use a regular expression which ignores any number of 25s after the
     *     trailing % and before the closing {@link #TOKEN_SUFFIX}</li>
     *     <li><i>I want to encode by depth</i>: Use the index of {@link #TOKEN_PREFIX} + this key to find the first
     *     25, if any are present, and the location of the first {@link #TOKEN_SUFFIX} to find the end. Use the number
     *     of 25s present to determine the depth</li>
     * </ol>
     */
    String BASE_URL_KEY = "stash.baseUrl%";
    /**
     * A prefix which is applied to the beginning of tokens which are emitted in built URLs.
     */
    String TOKEN_PREFIX = "$$$$";
    /**
     * A suffix which is appended to the ending of tokens which are emitted in built URLs.
     */
    String TOKEN_SUFFIX = "$$$$";
    /**
     * The fully-computed base URL token, comprised of {@link #TOKEN_PREFIX}, {@link #BASE_URL_KEY} and
     * {@link #TOKEN_SUFFIX}.
     */
    String BASE_URL_TOKEN = TOKEN_PREFIX + BASE_URL_KEY + TOKEN_SUFFIX;
    /**
     * The query parameter on the request which indicates the base URL on emitted URLs should be replaced with
     * {@link #BASE_URL_TOKEN} instead. Using this token allows callers to format in their own base URL for
     * producing links which are correct based on their context.
     * <p>
     * For example, to better explain the need for this, consider the following:
     * <ul>
     *     <li>A Jira instance is running with a public address of http://jira.example.com</li>
     *     <li>A Stash instance is running with a public address of http://stash.example.com</li>
     *     <li>These two machines, in addition to having public-facing URLs, have a private backplane network which
     *     connects them as jira.private.example.com and stash.private.example.com</li>
     *     <li>The application link from Jira to Stash has an RPC URL of http://stash.private.example.com, so that
     *     RPC uses the fast backplane network, and a display URL of http://stash.example.com so that end users
     *     get valid links</li>
     *     <li>When requests come into Stash, if the base URL calculated by {@link #buildAbsolute()} uses the HTTP
     *     request (which is the default behaviour), links are emitted pointing to http://stash.private.example.com.
     *     From an end user's browser, however, such links will not work</li>
     *     <li>Rather than trying to manually munge the URLs in the client, which is problematic and brittle, the
     *     caller can add this parameter to their request and a token suitable for replacement will be emitted in
     *     the URLs instead of a fixed base URL</li>
     * </ul>
     */
    String USE_BASE_URL_TOKEN = "useBaseUrlToken";

    /**
     * This method respects the {@link #USE_BASE_URL_TOKEN} parameter if present on the context request. Use
     * {@link #buildBaseUrl} if you with to avoid this behaviour. Note: if the {@link #USE_BASE_URL_TOKEN} is present
     * this method may return a {@link String} that is not a syntactically valid URI. The base URL (the scheme, host,
     * port etc and webapp context) will be taken from the context HTTP request, if one is present, or otherwise fall
     * back to the {@link ApplicationPropertiesService#getBaseUrl() configured base URL} for
     * the server.
     *
     * @return the absolute base url (e.g. http://hostname:7990/context), with no trailing slash
     * @see #buildBaseUrl
     * @see #USE_BASE_URL_TOKEN
     * @see #buildRelative()
     */
    String buildAbsolute();

    /**
     * This method is equivalent to {@link #buildAbsolute()}, but ignores any context HTTP request and always uses the
     * server's {@link ApplicationPropertiesService#getBaseUrl() configured base URL} for
     * generating the scheme, host, port etc and webapp context components of the URL.
     *
     * @return the absolute base url {@link ApplicationPropertiesService#getBaseUrl()
     * configured} for this server
     */
    String buildConfigured();

    /**
     * This method does not respect the {@link #USE_BASE_URL_TOKEN} parameter. Unless you have a specific need to avoid
     * the token substitution behaviour for this particular URL, you should use {@link #buildAbsolute()}.
     *
     * @return the absolute base url (e.g. http://hostname:7990/context), with no trailing slash
     * @see #buildAbsolute()
     * @see #USE_BASE_URL_TOKEN
     * @see #buildRelative()
     */
    String buildBaseUrl();

    /**
     * @return the relative base url (e.g. /context), with no trailing slash
     * @see #buildAbsolute()
     * @see #buildBaseUrl
     */
    String buildRelative();

    /** Base interface for all terminal builders (builders which can produce a url) */
    interface Builder<B extends Builder<B>> {

        /**
         * @param name the name of the parameter
         * @param value the value of the parameter
         * @return a builder with the supplied name and value parameters. Overwrites any previous parameter of the same name.
         */
        B withParam(@Nonnull String name, String value);

        /**
         * @return the url path without scheme, host, port etc e.g. /webappcontext/rest/of/url
         */
        String buildRelative();

        /**
         * @return the url path without scheme, host, port etc or webapp context e.g. /rest/of/url
         */
        String buildRelNoContext();

        /**
         * @return the full url including scheme, host, port etc and webapp context e.g. http://localhost:7990/bitbucket/rest/of/url
         * all components up to and including the webapp context will be taken from the context HTTP request (if one is present)
         * or otherwise taken the configured base URL of the server.
         **/
        String buildAbsolute();

        /**
         * @return the same as {@link #buildAbsolute()}, but will ignore any context HTTP request, and always use the
         * server's configured base URL
         **/
        String buildConfigured();
    }

    /**
     * A builder for urls that represent a collection of things, one page at a time.
     * Provides built-in support for the 'start' and 'limit' params.
     */
    interface PagedBuilder<B extends PagedBuilder<B>> extends Builder<B> {
        PagedBuilder<B> startAt(int index);
        PagedBuilder<B> withLimit(int limit);
    }

    /** The builder of urls that have a path to a file resource */
    interface PathBuilder<B extends PathBuilder<B>> extends Builder<B> {

        /**
         * @param path var args of path elements
         * @return a builder with the file resource path set as supplied
         */
        PathBuilder<B> path(@Nonnull String... path);

        /**
         * @param path the path
         * @return a builder with the file resource path set as supplied
         */
        PathBuilder<B> path(@Nonnull Path path);
    }

    Admin admin();
    Builder<?> about();
    PluginServlets pluginServlets();
    Rest rest();

    /**
     * /tmp
     * <p>
     * Retrieve a builder for constructing URLs related to temporary data.
     *
     * @return a builder for temporary data URLs
     */
    Tmp tmp();

    /**
     * @return a builder for UPM created URLs
     */
    Addons addons();

    /**
     * /projects/PROJ
     *
     * @param key project key
     * @return a builder for urls of the named project
     */
    Project project(@Nonnull String key);

    /**
     * /projects/PROJ
     *
     * @param project  the project
     * @return a builder for urls of the named project
     */
    Project project(@Nonnull com.atlassian.bitbucket.project.Project project);

    /**
     * /projects/PROJ/repos/REPO
     *
     * @param repository the repository
     * @return a builder for urls of the named repository
     */
    Repo repo(@Nonnull Repository repository);

    /**
     * /login
     *
     * @return a Login builder for the url to the login page
     */
    Login login();

    /**
     * /login
     *
     * @return a builder for the url to the logout page
     */
    Builder<?> logout();

    /**
     * /signup
     *
     * @return a builder for the url to the signup page
     */
    Builder<?> signup();

    /**
     * /captcha
     *
     * @return a builder for the url to the captcha image
     */
    Builder<?> captcha();

    /**
     * /retrievePassword
     *
     * @return a builder for the url to the page to retrieve your password
     */
    Builder<?> retrievePassword();

    /**
     * /passwordreset
     *
     * @return a builder for the url to the page to reset your password
     */
    Builder<?> passwordReset();

    /**
     * /profile
     *
     * @return a builder for the url to the profile page of the current user
     */
    Profile profile();

    /**
     * /users/SLUG or /bots/SLUG
     *
     * @param user the user
     * @return a builder for the url to the profile page of the supplied user
     */
    Profile user(ApplicationUser user);

    /**
     * /users/SLUG or /bots/SLUG
     *
     * @param userSlug the user slug
     * @param type the user type
     * @return a builder for the url to the profile page of the supplied user's slug
     */
    Profile userBySlug(String userSlug, UserType type);

    /**
     * /j_atl_security_check
     *
     * @return a builder for the url used to check authentication credentials when logging in
     */
    Builder<?> securityCheck();

    /**
     * /mvc/xsrfNotification (as this is generated by a forward it doesn't go through the rewrite filter, and so needs the /mvc
     * explicitly)
     *
     * @return a builder for the url to use when an invalid xsrf token is detected.
     */
    Builder<?> xsrfNotification();

    /**
     * /setup
     *
     * @return a builder for the first run wizard
     */
    Builder<?> setup();

    /**
     * /projects
     *
     * @return a builder for the url to list all projects
     */
    ListProjects allProjects();

    /**
     * /repos
     *
     * @return a builder for the url to list all repositories
     */
    GlobalListRepos allRepos();

    /** @return a builder for urls of the form /projects?create */
    CreateProject createProject();

    /** @return a builder for urls of the form /account[/*] */
    UserAccount account();

    /**
     * /getting-started
     *
     * @return a builder for urls to the Getting Started '/getting-started' page
     */
    GettingStarted gettingStarted();

    Dashboard dashboard();

    interface Dashboard extends Builder<Dashboard> { }

    interface Profile extends Builder<Profile> {
        /**
         * @param size the size for the avatar, in pixels
         * @return a builder for constructing avatar URLs
         */
        Builder<?> avatar(int size);
        Builder<?> listSshKeys();
    }

    /** The builder for /account */
    interface UserAccount extends Builder<UserAccount> {
        Builder<?> settings();
        Builder<?> password();
    }

    /** The builder for /admin */
    interface Admin extends Builder<Admin> {
        Users users();
        Groups groups();
        Builder<?> security();
        Builder<?> plugins();
        Builder<?> settings();
        Builder<?> permissions();
        Builder<?> avatars();
        Builder<?> authentication();
        Builder<?> mailServer();
        Builder<?> appLinks();
        Builder<?> serverSettings();
        Database database();
        Builder<?> storage();
        Licensing licensing();
        Builder<?> i18n();
        Builder<?> logging();
        Builder<?> sysInfo();
        Builder<?> clustering();
        /**
         * @since 6.6
         */
        Builder<?> rateLimitSettings();
        /**
         * @since 6.7
         */
        Builder<?> mirrors();
        /**
         * @since 7.5
         */
        GitAdmin git();
        /**
         * @since 7.8
         */
        JiraCloudIntegration jiraCloudIntegration();
        /**
         * @since 7.12
         */
        Builder<?> repositoryManagement();
        /**
         * @since 8.3
         */
        Builder<?> secretScanning();
        /**
         * @since 8.7
         */
        KeysAndTokens keysAndTokens();
    }

    interface Database extends Builder<Database> {
        Builder<?> migrate();
    }

    /**
     * @since 7.5
     */
    interface GitAdmin extends Builder<GitAdmin> {
        Builder<?> defaultBranch();

        /**
         * @return a builder for URLs of the form {@code /admin/git/mesh}
         * @since 7.12
         */
        Mesh mesh();
    }

    /**
     * The builder for the JSW Admin page.
     *
     * @since 7.8
     */
    interface JiraCloudIntegration extends Builder<JiraCloudIntegration> {
        Builder<?> jiraCloud();
    }

    interface Licensing extends Builder<Licensing> {
        Builder<?> edit();
    }

    /**
     * A builder for URLs at/under {@code /admin/git/mesh}.
     *
     * @since 7.12
     */
    interface Mesh extends Builder<Mesh> {
    }

    /**
     * A builder for URLs at/under {@code /admin/keys-and-tokens}.
     *
     * @since 8.7
     */
    interface KeysAndTokens extends Builder<KeysAndTokens> {
    }

    interface Users extends PagedBuilder<Users> {
        /**
         * @return {@code this}
         * @since 5.16
         */
        Builder<?> anonymize();
        Builder<?> create();
        DeleteUser delete(String username);
        ViewUser view(String username);
        Users filter(String filter); // list of users whose names, display names or email addresses contains 'filter'
        Builder<?> captcha(String username); // the current captcha challenge placed on a user
    }

    /** The builder for /users/view */
    interface ViewUser extends Builder<ViewUser> {
        Builder<?> listSshKeys();
    }

    /** The builder for /users/delete */
    interface DeleteUser extends Builder<DeleteUser> { }

    interface Groups extends PagedBuilder<Groups> {
        Builder<?> create();
        DeleteGroup delete(String groupName);
        ViewGroup view(String name);
        Groups filter(String filter); // list of groups whose names contains 'filter'
    }

    interface DeleteGroup extends Builder<DeleteGroup> { }

    /** The builder for /group/view */
    interface ViewGroup extends Builder<ViewGroup> { }

    /** The builder for the plugin servlets */
    interface PluginServlets extends PathBuilder<PluginServlets> {
        Builder<?> oauthTokens();
        Builder<?> notificationSettings();
        Builder<?> repositoryNotificationSettings();
    }

    /** The builder for the REST API */
    interface Rest extends Builder<Rest> {
        RestAdmin admin();
        RestUsers users();
        RestUser users(String slug);
        Builder<?> pullRequest(long id);
        RestHooks hooks();
    }

    /**
     * The builder of urls of the form /rest/hooks
     */
    interface RestHooks extends Builder<RestHooks> {
        RestHook hook(@Nonnull String hookKey);
    }

    /**
     * The builder of urls of the form /rest/hooks/KEY
     */
    interface RestHook extends Builder<RestHook> {
        Builder<?> avatar(@Nullable String version);
    }

    /**
     * The builder for /rest/users
     */
    interface RestUsers extends Builder<RestUsers> { }

    /**
     * The builder for /rest/users/slug
     */
    interface RestUser extends Builder<RestUser> {
        Builder<?> avatar();
    }

    /**
     * The builder for the REST admin API
     */
    interface RestAdmin extends Builder<RestAdmin> {
        Builder<?> users();
    }

    /**
     * The builder for /projects
     */
    interface ListProjects extends PagedBuilder<ListProjects> { }

    /**
     * The builder for /repos
     */
    interface GlobalListRepos extends Builder<GlobalListRepos> {

        /**
         * The builder for /repos?visibility=private. Convenient method when invoking in velocity
         * @return list of private repositories
         */
        GlobalListRepos privateVisibility();

        /**
         * The builder for /repos?visibility=public. Convenient method when invoking in velocity
         * @return list of public repositories
         */
        GlobalListRepos publicVisibility();

        /**
         * The builder for /repos?visibility=[level]
         * @param visibility visibility
         * @return list of repositories with the given visibility
         */
        GlobalListRepos visibility(RepositoryVisibility visibility);

    }

    /**
     * The builder for urls hanging off /projects/PROJ. The url path built by this builder, if not further developed, is the same as {@code ListRepos}
     */
    interface Project extends Builder<Project> {

        Builder<?> avatar(int size);

        /**
         * @return a builder for urls of the form /projects/PROJ/settings/hooks
         */
        ProjectHooks hooks();

        /**
         * @return a builder for urls of the form /projects/PROJ/settings/merge-checks
         */
        ProjectMergeChecks mergeChecks();

        /**
         * @return a builder for urls of the form /projects/PROJ - essentially the same as {@link NavBuilder.Project}
         * but with support for start and limit params to control the repository list
         */
        ListRepos repos();

        /**
         * @param slug the repository slug
         * @return a builder for urls of the form /projects/PROJ/repos/REPO[/...]
         */
        Repo repo(@Nonnull String slug);

        /**
         * @param repository the repository
         * @return a builder for urls of the form /projects/PROJ/repos/REPO[/...]
         */
        Repo repo(@Nonnull Repository repository);

        /**
         * @return a builder for urls of the form /projects/PROJ/repos?create
         */
        CreateRepo createRepo();

        /**
         * @return a builder for urls of the form /projects/PROJ/settings
         */
        ProjectSettings settings();

        /**
         * @return a builder for urls of the form /projects/PROJ
         */
        ProjectDelete delete();

        /**
         * @return a builder for urls of the form /projects/PROJ/permission
         */
        ProjectPermissions permissions();

        /**
         * @return a builder of urls of the form /project/PROJ/settings/secret-scanning
         * @since 8.5
         */
        ProjectSecretScanning secretScanning();
    }

    /**
     * The builder for /projects/PROJ/repos/REPO/browse - essentially the same as {@link NavBuilder.Project} but with
     * support for start and limit params o control the repository list
     */
    interface ListRepos extends PagedBuilder<ListRepos> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/hooks
     */
    interface ProjectHooks extends Builder<ProjectHooks> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/merge-checks
     */
    interface ProjectMergeChecks extends Builder<ProjectMergeChecks> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings
     */
    interface ProjectSettings extends Builder<ProjectSettings> {

        /**
         * @return the auto decline settings
         * @since 7.7
         */
        ProjectAutoDeclineSettings autoDecline();

        /**
         * @return the default tasks settings
         * @since 8.4
         */
        ProjectDefaultTasksSettings defaultTasks();

        /**
         * @param scmId the scmId
         * @return the merge strategy settings
         * @since 5.2
         */
        ProjectMergeStrategySettings mergeStrategies(@Nonnull String scmId);

        /**
         * @return a builder for urls of the form /projects/PROJ/settings/pull-request-templates
         * @since 7.13
         */
        ProjectPullRequestTemplateSettings pullRequestTemplates();

        /**
         * @return the reviewer groups settings
         * @since 7.13
         */
        ProjectReviewerGroupSettings reviewerGroups();
    }

    /**
     * The builder of urls of the form /projects/PROJ?create
     */
    interface CreateProject extends Builder<CreateProject> { }

    /**
     * The builder of urls of the form /projects/PROJ/delete
     */
    interface ProjectDelete extends Builder<ProjectDelete> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/auto-decline
     *
     * @since 7.7
     */
    interface ProjectAutoDeclineSettings extends Builder<ProjectAutoDeclineSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/default-tasks
     *
     * @since 8.4
     */
    interface ProjectDefaultTasksSettings extends Builder<ProjectDefaultTasksSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/merge-strategies/&lt;scmId&gt;
     *
     * @since 5.2
     */
    interface ProjectMergeStrategySettings extends Builder<ProjectMergeStrategySettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/pull-request-templates
     * @since 7.13
     */
    interface ProjectPullRequestTemplateSettings extends Builder<ProjectPullRequestTemplateSettings> {}

    /**
     * The builder of urls of the form /projects/PROJ/permissions
     */
    interface ProjectPermissions extends Builder<ProjectPermissions> { }

    /**
     * The builder of urls of the form /projects/PROJ/settings/reviewer-groups
     * @since 7.13
     */
    interface ProjectReviewerGroupSettings extends Builder<ProjectReviewerGroupSettings> { }

    /**
     * The builder of urls for the form /project/PROJ/settings/secret-scanning
     * @since 8.5
     */
    interface ProjectSecretScanning extends Builder<ProjectSecretScanning> { }

    /**
     * The builder of urls hanging off /projects/PROJ/repos/REPO. The url built by this builder, if not further developed,
     * is the same as {@code BrowseRepo}
     */
    interface Repo extends Builder<Repo> {

        /**
         * @return a builder for URLs to download archives of a repository
         * @since 5.1
         */
        @Nonnull
        RepoArchive archive();

        /**
         * Specifically used for legacy attachments.
         *
         * @param id the attachment ID
         * @return a builder for the URL of the form /projects/PROJ/repos/REPO/attachments/ID
         * @since 7.0
         */
        @Nonnull
        RepoAttachment attachment(@Nonnull String id);

        /**
         * Specifically used for modern attachments.
         *
         * @param id the attachment ID
         * @return a builder for the URL of the form /projects/PROJ/repos/REPO/attachments/ID
         * @since 7.0
         */
        @Nonnull
        RepoAttachment attachment(long id);

        /** @return a builder for urls of the form /projects/PROJ/repos/REPO/browse[/...] */
        @Nonnull
        BrowseRepoResource browse();

        /** @return a builder for urls of the form /projects/PROJ/repos/REPO/branches */
        @Nonnull
        ListBranches branches();

        /**
         * @return a builder for URls of the form /projects/PROJ/repos/REPO/builds
         * @since 7.4
         */
        @Nonnull
        RepoBuilds builds();

        /** @return a builder for urls of the form /projects/PROJ/repos/REPO/tags */
        @Nonnull
        ListTags tags();

        /**
         * @param commitId the {@link com.atlassian.bitbucket.commit.Commit#getId commit ID} to build URLs for
         * @return a builder for URLs of the form /projects/PROJ/repos/REPO/commits/ID
         */
        @Nonnull
        Commit commit(@Nonnull String commitId);

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/commits
         */
        @Nonnull
        ListCommits commits();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/compare
         */
        @Nonnull
        Compare compare();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/forks
         * @since 5.4
         */
        @Nonnull
        ListForks forks();

        /** @return a builder for urls of the form /projects/PROJ/repos/REPO/settings */
        @Nonnull
        RepoSettings settings();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/settings/hooks
         */
        @Nonnull
        RepoHooks hooks();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/settings/merge-checks
         */
        @Nonnull
        RepoMergeChecks mergeChecks();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/permissions
         */
        @Nonnull
        RepoPermissions permissions();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO
         */
        @Nonnull
        RepoDelete delete();

        @Nonnull
        RepoFork fork();

        /**
         * @return a builder for urls of the form /projects/PROJ/repos/REPO/push-log
         * @since 7.1
         */
        @Nonnull
        RepoPushLog pushLog();

        @Nonnull
        CreatePullRequest createPullRequest();

        @Nonnull
        PullRequest pullRequest(long id);

        @Nonnull
        ListPullRequests pullRequests();
    }

    /** The builder of urls of the form /projects/PROJ/repos/REPO/browse[/...] */
    interface BrowseRepoResource extends PathBuilder<BrowseRepoResource> {

        /**
         * The builder for urls of the form /projects/PROJ/repos/REPO/browse/...?at=REVISION
         *
         * @param revision revision to browse at
         * @return revision browse
         */
        RevisionBrowse atRevision(@Nullable String revision);

        /**
         * @param ref the ref to browse at
         * @return revision browse
         */
        RevisionBrowse atRevision(@Nonnull Ref ref);
    }

    /**
     * The builder of urls of the form /projects/PROJ/repos?create
     */
    interface CreateRepo extends Builder<CreateRepo> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/branches
     */
    interface ListBranches extends PagedBuilder<ListBranches> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/tags[/...]
     */
    interface ListTags extends PagedBuilder<ListTags> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/commits
     */
    interface ListCommits extends PagedBuilder<ListCommits> {
        ListCommits until(@Nonnull Ref ref);
        ListCommits until(@Nonnull String until);
    }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/forks
     * @since 5.4
     */
    interface ListForks extends PagedBuilder<ListForks> { }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/commits/ID
     */
    interface Commit extends PathBuilder<Commit> {

        /**
         * Adds an anchor for the specified path, producing a URL like /projects/PROJ/repos/repo/commits/ID#path.
         *
         * @param path the path to set in the anchor
         * @return {@code this}
         */
        @Nonnull
        Commit change(@Nonnull Path path);

        /**
         * Adds an anchor for the specified path, producing a URL like /projects/PROJ/repos/repo/commits/ID#path.
         *
         * @param path the path to set in the anchor
         * @return {@code this}
         */
        @Nonnull
        Commit change(@Nonnull String path);

        /**
         * Retrieves a builder for interacting with the specified comment on this commit.
         *
         * @param commentId the ID of the comment to build URLs for
         * @return a comment URL builder
         */
        @Nonnull
        CommitComment comment(long commentId);

        /**
         * Adds an {@code unwatch} query parameter to the URL for this commit.
         *
         * @return {@code this}
         */
        @Nonnull
        Commit unwatch();
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/commits/ID/comment/ID
     */
    interface CommitComment extends Builder<CommitComment> {
    }

    /**
     * Builds URLs of the form /rest/api/1.0/projects/PROJ/repos/REPO/archive
     *
     * @since 5.1
     */
    interface RepoArchive extends Builder<RepoArchive> {

        @Nonnull
        RepoArchive commitId(@Nonnull String value);

        @Nonnull
        RepoArchive filename(@Nonnull String value);

        @Nonnull
        RepoArchive format(@Nonnull ArchiveFormat value);

        @Nonnull
        RepoArchive path(@Nullable String value);

        @Nonnull
        RepoArchive paths(@Nullable String value, @Nullable String... values);

        @Nonnull
        RepoArchive paths(@Nullable Iterable<String> values);

        @Nonnull
        RepoArchive prefix(@Nonnull String value);
    }

    /**
     * The builder for URLs of the form /projects/PROJ/repos/REPO/attachments
     *
     * @since 7.0
     */
    interface RepoAttachment extends Builder<RepoAttachment> {

        /**
         * @return an attachment metadata builder
         * @throws UnsupportedOperationException when used for a legacy attachment
         */
        @Nonnull
        RepoAttachmentMetadata metadata();
    }

    /**
     * The builder for the URLs of the form /projects/PROJ/repos/REPO/attachments/ID/metadata
     *
     * @since 7.0
     */
    interface RepoAttachmentMetadata extends Builder<RepoAttachmentMetadata> {
    }

    /**
     * The builder for urls of the form /projects/PROJ/repos/REPO/builds
     *
     * @since 7.4
     */
    interface RepoBuilds extends Builder<RepoBuilds> {

        /**
         * The builder for urls of the form /projects/PROJ/repos/REPO/builds/?at=REVISION
         *
         * @param revision the revision to display builds for
         * @return branch builds
         */
        RevisionBuilds atRevision(@Nullable String revision);

        /**
         * The builder for urls of the form /projects/PROJ/repos/REPO/builds/?at=REVISION
         *
         * @param ref the ref to display builds for
         * @return branch builds
         */
        RevisionBuilds atRevision(@Nonnull Ref ref);
    }

    /**
     * The builder for urls of the form /projects/PROJ/repos/REPO/builds?at=REVISION
     */
    interface RevisionBuilds extends PathBuilder<RevisionBuilds> { }


    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/compare
     */
    interface Compare extends CompareBuilder<Compare>, PathBuilder<Compare> {
        CommitsBuilder commits();
        DiffBuilder diff();

        interface CommitsBuilder extends CompareBuilder<CommitsBuilder> { }
        interface DiffBuilder extends CompareBuilder<DiffBuilder> { }
    }

    interface CompareBuilder<T extends CompareBuilder<T>> extends Builder<T> {
        CompareBuilder<T> sourceBranch(@Nonnull String sourceBranch);
        CompareBuilder<T> targetBranch(@Nonnull String targetBranch);
    }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings
     */
    interface RepoSettings extends Builder<RepoSettings> {
        /**
         * @return the auto decline settings
         * @since 7.7
         */
        RepoAutoDeclineSettings autoDecline();

        /**
         * @return the default tasks settings
         * @since 8.4
         */
        RepoDefaultTasksSettings defaultTasks();

        /**
         * @return the pull request settings
         * @since 2.3
         */
        PullRequestRepoSettings pullRequests();

        /**
         * @return the merge strategies
         * @since 5.2
         */
        RepoMergeStrategySettings mergeStrategies();

        /**
         * @return the pull request templates settings
         * @since 7.16
         */
        RepoPullRequestTemplateSettings pullRequestTemplates();

        /**
         * @return the required builds
         * @since 7.14
         */
        RepoRequiredBuildsSettings requiredBuilds();

        /**
         * @return the reviewer groups settings
         * @since 7.13
         */
        RepoReviewerGroupsSettings reviewerGroups();

        /**
         * @return the secret scanning settings
         * @since 8.5
         */
        RepoSecretScanning secretScanning();
    }

    interface RepoPermissions extends Builder<RepoPermissions> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/pull-requests
     */
    interface PullRequestRepoSettings extends Builder<PullRequestRepoSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/hooks
     */
    interface RepoHooks extends Builder<RepoHooks> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/merge-checks
     */
    interface RepoMergeChecks extends Builder<RepoMergeChecks> { }

    /** The builder of urls of the form /projects/PROJ/repos/REPO/delete */
    interface RepoDelete extends Builder<RepoDelete> { }

    /** The builder for urls of the form /projects/PROJ/repos/REPO/browse/...?at=REVISION */
    interface RevisionBrowse extends PathBuilder<RevisionBrowse> { }

    /** The builder for urls of the form /projects/PROJ/repos/REPO?fork
     */
    interface RepoFork extends Builder<RepoFork> { }

    /** The builder for urls of the form /projects/PROJ/repos/REPO/push-log
     */
    interface RepoPushLog extends Builder<RepoPushLog> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/auto-decline
     *
     * @since 7.7
     */
    interface RepoAutoDeclineSettings extends Builder<RepoAutoDeclineSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/default-tasks
     *
     * @since 8.4
     */
    interface RepoDefaultTasksSettings extends Builder<RepoDefaultTasksSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/merge-strategies
     *
     * @since 5.2
     */
    interface RepoMergeStrategySettings extends Builder<RepoMergeStrategySettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/pull-request-templates
     * @since 7.16
     */
    interface RepoPullRequestTemplateSettings extends Builder<RepoPullRequestTemplateSettings> {}

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/required-builds
     *
     * @since 7.14
     */
    interface RepoRequiredBuildsSettings extends Builder<RepoRequiredBuildsSettings> { }

    /**
     * The builder of urls of the form /projects/PROJ/repos/REPO/settings/reviewer-groups
     *
     * @since 7.13
     */
    interface RepoReviewerGroupsSettings extends Builder<RepoReviewerGroupsSettings> { }

    /**
     * The builder of urls for the form /project/PROJ/repos/REPO/settings/secret-scanning
     * @since 8.5
     */
    interface RepoSecretScanning extends Builder<RepoSecretScanning> { }

    /** The builder for urls of the form /projects/PROJ/repos/REPO/pull-requests?create... */
    interface CreatePullRequest extends Builder<CreatePullRequest> {
        CreatePullRequest sourceBranch(String sourceBranch);
        CreatePullRequest targetBranch(String targetBranch);
    }

    /** The builder for urls of the form /projects/PROJ/repos/REPO/pull-requests/ID/... */
    interface PullRequest extends Builder<PullRequest> {

        /**
         * View the activities of the pull request.
         *
         * @param activityId The unique identifier of the activity
         * @return {@link NavBuilder.PullRequestActivity} a builder used to construct URLs for activities
         */
        @Nonnull
        PullRequestActivity activity(long activityId);

        /**
         * View the builds for that pull request.
         *
         * @return a {@link PullRequestBuilds} builder used to construct URLs for builds
         */
        @Nonnull
        PullRequestBuilds builds();

        /**
         * View the comments added to the pull request.
         *
         * @param commentId The unique identifier of the comment
         * @return a {@link PullRequestComment} builder used to construct URLs for comments
         */
        @Nonnull
        PullRequestComment comment(long commentId);

        /**
         * View a specific commit within a pull request.
         * <p>
         * Unlike {@link Repo#commit(String) repository commits}, pull request commits <i>must</i> be specified by
         * {@link com.atlassian.bitbucket.commit.Commit#getId() ID}. Ref names like "master" do not make sense in this
         * context.
         *
         * @param commitId the {@link com.atlassian.bitbucket.commit.Commit#getId commit ID} to build URLs for
         * @return a builder which allows deep linking to a specific path within the commit
         */
        @Nonnull
        PullRequestCommit commit(@Nonnull String commitId);

        /**
         * View the commits to be merged by a pull request.
         *
         * @return a builder which allows linking to a page of commits
         */
        PagedBuilder<?> commits();

        /**
         * View the pull request's diff.
         *
         * @return a builder which allows linking to the diff
         */
        Builder<?> diff();

        /**
         * View the pull request's overview, which displays snippets for activities which have been performed on the
         * pull request as well its description.
         *
         * @return a builder which allows linking to the overview
         */
        PullRequestOverview overview();

        /**
         * View the pull request diff, and the unreviewed changes for the current user, if they exist.
         *
         * @return a builder which allows linking to the diff of the unreviewed changes
         * @since 4.10
         */
        Builder<?> unreviewed();

        /**
         * Remove the current user as a watcher for the pull request.
         *
         * @return a builder which allows building links to stop watching the pull request
         */
        PagedBuilder<?> unwatch();
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/pull-requests/ID/activities
     */
    interface PullRequestActivity extends Builder<PullRequestActivity> {
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/pull-requests/ID/builds
     */
    interface PullRequestBuilds extends Builder<PullRequestBuilds> {
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/pull-requests/ID/overview?commentId=ID
     */
    interface PullRequestComment extends Builder<PullRequestComment> {
        /**
         * Deep link to liking a pull request comment.
         * @return {@link NavBuilder.PullRequestComment} a builder,
         * which is capable of constructing a URL to like a comment
         */
        @Nonnull
        PullRequestComment like();
        /**
         * Deep link to replying to a pull request comment
         * @return {@link NavBuilder.PullRequestComment} a builder,
         * which is capable of constructing a URL to reply to a comment
         */
        @Nonnull
        PullRequestComment reply();
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/pull-requests/ID/commits/ID
     */
    interface PullRequestCommit extends PathBuilder<PullRequestCommit> {

        PullRequestCommit since(@Nonnull String sinceId);
    }

    /**
     * Builds URLs of the form /projects/PROJ/repos/REPO/pull-requests/ID/overview
     */
    interface PullRequestOverview extends PagedBuilder<PullRequestOverview> {
    }

    interface ListPullRequests extends PagedBuilder<ListPullRequests> {
        Builder<?> open();
        Builder<?> declined();
        Builder<?> merged();
        ListPullRequests incoming();
        ListPullRequests outgoing();
    }

    /**
     * Group builder for endpoints under /tmp, which are intended to provide access to temporary data stored by the
     * system to facilitate user actions.
     */
    interface Tmp extends Builder<Tmp> {

        /**
         * Builds URLs for accessing temporary avatars which have been uploaded. When a temporary avatar is uploaded,
         * a unique ID is generated and returned. That ID is a required parameter here.
         *
         * @param id the ID assigned when the temporary avatar was uploaded
         * @return a builder to construct URLs for accessing the specified temporary avatar
         */
        Builder<?> avatar(@Nonnull String id);
    }

    interface Addons extends Builder<Addons> {

        Builder<?> manage();

        AddonMarketplace marketplace();

        AddonRequests requests();
    }

    interface AddonMarketplace extends Builder<AddonMarketplace> {
        Builder<?> category(String category);
    }

    interface AddonRequests extends Builder<AddonRequests> {
        Builder<?> category(String category);
    }

    /**
     * Builds URL for the with an option of adding "next" as a query parameter with the current request url
     * or the passed in url
     */
    interface NavigationStep<B extends NavigationStep<B>> extends Builder<B> {

        /**
         * Add "next" as a query parameter with the current relative request url as it value.
         *
         * @return this builder instance
         */
        @Nonnull
        B next();

        /**
         * Add "next" as a query parameter with url as the value.
         *
         * @param url the url to add as the {@code next} query parameter
         * @return this builder instance
         */
        @Nonnull
        B next(@Nonnull String url);
    }

    /**
     * Builds a login URL for the with an option of adding "next" as a query parameter with the current request url
     * or the passed in url
     */
    interface Login extends NavigationStep<Login> {

        /**
         * Return a login URL that points to the embedded version of the login form, suitable to display in a frame in
         * other applications.
         *
         * @return URL to the embedded login form
         */
        @Nonnull
        Login embedded();

        /**
         * Return an URL that will open a login page requiring the
         * {@link Permission#ADMIN admin permission} to log-in.
         *
         * @return login form guarding a resource available to administrators only
         */
        @Nonnull
        Login admin();

        /**
         * Return an URL that will open a login page requiring the
         * {@link Permission#ADMIN admin permission} to log-in.
         *
         * @return login form guarding a resource available to administrators only
         */
        @Nonnull
        Login sysadmin();
    }

    /**
     * Builds a URL to the GettingStarted page with an option of adding "next" as a query parameter with the current
     * request url or the passed in url
     */
    interface GettingStarted extends NavigationStep<GettingStarted> {
    }
}
