package com.atlassian.bitbucket.job;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;

import static java.util.Optional.ofNullable;

/**
 * A request to find {@link JobMessage job messages}
 *
 * @since 5.13
 */
public class JobMessageSearchRequest {

    private final Job job;
    private final Set<JobMessageSeverity> severities;
    private final String subject;

    private JobMessageSearchRequest(Builder builder) {
        job = builder.job;
        severities = ImmutableSet.copyOf(builder.severities);
        subject = builder.subject;
    }

    @Nonnull
    public Optional<Job> getJob() {
        return ofNullable(job);
    }

    @Nonnull
    public Set<JobMessageSeverity> getSeverities() {
        return severities;
    }

    @Nonnull
    public Optional<String> getSubject() {
        return ofNullable(subject);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JobMessageSearchRequest that = (JobMessageSearchRequest) o;
        return Objects.equal(getJob().map(Job::getId), that.getJob().map(Job::getId)) &&
                Objects.equal(getSeverities(), that.getSeverities()) &&
                Objects.equal(getSubject(), that.getSubject());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getJob(), getSeverities(), getSubject());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("job.id", job.getId())
                .add("severities", severities)
                .add("subject", subject)
                .toString();
    }

    public static class Builder extends BuilderSupport {

        private Job job;
        private EnumSet<JobMessageSeverity> severities;
        private String subject;

        public Builder() {
            severities = EnumSet.noneOf(JobMessageSeverity.class);
        }

        @Nonnull
        public JobMessageSearchRequest build() {
            return new JobMessageSearchRequest(this);
        }

        @Nonnull
        public Builder job(@Nullable Job value) {
            job = value;
            return this;
        }

        @Nonnull
        public Builder severity(@Nullable JobMessageSeverity value) {
            addIf(java.util.Objects::nonNull, severities, value);
            return this;
        }

        @Nonnull
        public Builder severities(@Nonnull Iterable<JobMessageSeverity> values) {
            addIf(java.util.Objects::nonNull, severities, values);
            return this;
        }

        @Nonnull
        public Builder subject(@Nullable String value) {
            subject = value;
            return this;
        }
    }
}
