package com.atlassian.bitbucket.hook.script;

import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * @since 6.2
 */
public class HookScriptCreateRequest {

    private final InputSupplier<InputStream> content;
    private final String description;
    private final String name;
    private final String pluginKey;
    private final HookScriptType type;

    private HookScriptCreateRequest(Builder builder) {
        content = builder.content;
        description = builder.description;
        name = builder.name;
        pluginKey = builder.pluginKey;
        type = builder.type;
    }

    @Nonnull
    public InputSupplier<InputStream> getContent() {
        return content;
    }

    @Nonnull
    public Optional<String> getDescription() {
        return ofNullable(description);
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nonnull
    public String getPluginKey() {
        return pluginKey;
    }

    @Nonnull
    public HookScriptType getType() {
        return type;
    }

    public static class Builder extends BuilderSupport {

        private final String name;
        private final String pluginKey;
        private final HookScriptType type;

        private InputSupplier<InputStream> content;
        private String description;

        public Builder(@Nonnull String name, @Nonnull String pluginKey, @Nonnull HookScriptType type) {
            this.name = requireNonBlank(name, "name");
            this.pluginKey = requireNonBlank(pluginKey, "pluginKey");
            this.type = requireNonNull(type, "type");
        }

        @Nonnull
        public HookScriptCreateRequest build() {
            if (content == null) {
                throw new IllegalStateException("Contents must be provided when creating a new hook script");
            }

            return new HookScriptCreateRequest(this);
        }

        @Nonnull
        public Builder content(@Nonnull InputSupplier<InputStream> value) {
            content = requireNonNull(value, "content");

            return this;
        }

        @Nonnull
        public Builder content(@Nonnull String value) {
            //Hold onto the bytes, rather than the String
            return content(requireNonBlank(value, "content").getBytes(StandardCharsets.UTF_8));
        }

        @Nonnull
        public Builder content(@Nonnull byte[] value) {
            if (requireNonNull(value, "content").length == 0) {
                throw new IllegalArgumentException("Hook script contents cannot be empty");
            }
            content = () -> new ByteArrayInputStream(value);

            return this;
        }

        @Nonnull
        public Builder description(@Nullable String value) {
            description = trimToNull(value);

            return this;
        }
    }
}
