package com.atlassian.bitbucket.event.server;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.bitbucket.server.OperatingSystem;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Event raised after migrating between major operating systems.
 * <p>
 * <b>Note</b>: This event is delivered synchronously while the system is still starting. Handlers should be
 * as fast as possible to prevent excessive startup delays. If an app can lazily detect and recover from an
 * operating system migration, it's encouraged to <i>not</i> handle this event, or to use this event only to
 * trigger asynchronous processing to allow startup to continue.
 *
 * @since 6.2
 */
@EventName("stash.operating.system.changed")
public class OperatingSystemChangedEvent extends ApplicationEvent {

    private final OperatingSystem newValue;
    private final OperatingSystem oldValue;

    public OperatingSystemChangedEvent(@Nonnull Object source, @Nonnull OperatingSystem newValue,
                                       @Nonnull OperatingSystem oldValue) {
        super(source);
        this.newValue = requireNonNull(newValue, "newValue");
        this.oldValue = requireNonNull(oldValue, "oldValue");
    }

    @Nonnull
    public OperatingSystem getNewValue() {
        return newValue;
    }

    @Nonnull
    public OperatingSystem getOldValue() {
        return oldValue;
    }
}
