package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositoryForkRequest;
import com.atlassian.bitbucket.repository.RepositoryService;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Raised just before a {@link Repository repository} is forked. This event is synchronous, allowing listeners to
 * perform operations in the same database transaction where the forked repository will be created.
 * <p>
 * The {@link #getRepository() repository} attached to the event is the fork to be created. Its
 * {@link Repository#getOrigin() origin} provides the repository that will be forked to create it. Note that forks can
 * be created in the same project as their origin or in different one, so the {@link Repository#getProject() project}s
 * for the fork and its origin may or may not be the same.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the repository from being forked by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the repository
 * from being forked; the exception will be logged and ignored.
 * <p>
 * Note: The repository attached to the event will <i>not</i> have an {@link Repository#getId() ID}, as it has not
 * been created yet, but it <i>will</i> have its {@link Repository#getOrigin() origin repository} available. Any
 * listener which would like to create associations based on the repository ID must listen for the
 * {@link RepositoryForkedEvent} instead, which will include it (but will not run in the same transaction).
 *
 * @see RepositoryService#fork(RepositoryForkRequest)
 */
public class RepositoryForkRequestedEvent extends RepositoryCreationRequestedEvent {

    /**
     * @param source        the source of the event
     * @param repository    the fork to be created
     * @param defaultBranch the default branch for the fork, or {@code null} if the origin's default will be inherited
     * @param cancelState   tracks vetoes for the event
     * @since 7.5
     */
    public RepositoryForkRequestedEvent(@Nonnull Object source, @Nonnull Repository repository,
                                        @Nullable String defaultBranch, @Nonnull CancelState cancelState) {
        super(source, repository, defaultBranch, cancelState);
    }

    /**
     * Retrieves the default branch the fork will use, if one was specified. If no default was specified the
     * {@link Repository#getOrigin origin's} default branch will be inherited. This differs from creating a
     * new repository, where the SCM's default branch is used if one isn't specified.
     *
     * @return the default branch, or {@code null} if the origin's default branch will be inherited
     * @since 7.5
     */
    @Nullable
    @Override
    public String getDefaultBranch() {
        return super.getDefaultBranch();
    }
}
