package com.atlassian.bitbucket.event.pull;

import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.event.api.AsynchronousPreferred;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * An event raised when the total set of reviewers and role-less participants for a pull request changes.
 * <p>
 * This may happen for any of the following reasons in isolation or in combination:
 * <ul>
 * <li>A user is added as a reviewer for the first time. In this case they will appear in the Set returned
 * from {@link #getAddedParticipants()}</li>
 * <li>A user who is neither a reviewer, author nor role-less participant gives their +1 to, posts a comment to or edits
 * a pull request. In this case they will appear in the Set returned from {@link #getAddedParticipants()}</li>
 * <li>A reviewer is removed from the pull request and because, as reviewer, they never commented, edited or gave their
 * approval they have been completely removed from the pull request. In this case they will appear in the Set returned
 * from {@link #getRemovedParticipants()}
 * <p>Note: if a reviewer posts a comment, gives their approval or edits a pull request then any subsequent operation
 * to remove them as a reviewer would merely relegate them from a reviewer to a participant and this event will not be
 * fired for that change.</li>
 * </ul>
 * <p>
 * This event is internally audited with {@link CoverageLevel#BASE} coverage level
 */
@AsynchronousPreferred
public class PullRequestParticipantsUpdatedEvent extends PullRequestEvent {

    private final Set<ApplicationUser> addedParticipants;
    private final Set<ApplicationUser> removedParticipants;

    public PullRequestParticipantsUpdatedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                               @Nonnull Set<ApplicationUser> added, @Nonnull Set<ApplicationUser> removed) {
        super(source, pullRequest, PullRequestAction.UPDATED);

        this.addedParticipants = ImmutableSet.copyOf(added);
        this.removedParticipants = ImmutableSet.copyOf(removed);
    }

    /**
     * @return the set of users who have participated in the pull request for the first time
     */
    @Nonnull
    public Set<ApplicationUser> getAddedParticipants() {
        return addedParticipants;
    }

    /**
     * @return the set of users who are no longer participating in the pull request
     */
    @Nonnull
    public Set<ApplicationUser> getRemovedParticipants() {
        return removedParticipants;
    }
}
