package com.atlassian.bitbucket.event.mirror;

import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.bitbucket.event.repository.RepositoryEvent;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Base class for {@link Repository} events that have been raised on a mirror.
 * <p>
 * This class does not extend {@link RepositoryEvent} because events derived from this class relate to
 * mirrored repositories while the {@link RepositoryEvent} hierarchy relates to local repositories.
 *
 * @since 5.0
 */
public class AbstractRepositoryMirrorEvent extends ApplicationEvent implements RepositoryMirrorEvent {

    private final String mirrorServerId;
    private final String mirrorServerName;
    private final Repository repository;

    protected AbstractRepositoryMirrorEvent(Builder builder) {
        super(builder.source);
        this.mirrorServerId = builder.mirrorServerId;
        this.mirrorServerName = builder.mirrorServerName;
        this.repository = builder.repository;
    }

    @Nonnull
    @Override
    public String getMirrorServerId() {
        return mirrorServerId;
    }

    @Nonnull
    @Override
    public String getMirrorServerName() {
        return mirrorServerName;
    }

    @Override
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    protected abstract static class Builder<B extends Builder, T extends AbstractRepositoryMirrorEvent> {

        private String mirrorServerId;
        private String mirrorServerName;
        private Repository repository;
        private Object source;

        @Nonnull
        public abstract T build();

        @Nonnull
        public B mirrorServerId(@Nonnull String value) {
            mirrorServerId = requireNonNull(value, "mirrorServerId");
            return self();
        }

        @Nonnull
        public B mirrorServerName(@Nonnull String value) {
            mirrorServerName = requireNonNull(value, "mirrorServerName");
            return self();
        }

        @Nonnull
        public B repository(@Nonnull Repository value) {
            repository = requireNonNull(value, "repository");
            return self();
        }

        @Nonnull
        public B source(@Nonnull Object value) {
            source = requireNonNull(value, "source");
            return self();
        }

        @SuppressWarnings("unchecked")
        protected B self() {
            return (B) this;
        }

        protected void validate() {
            if (mirrorServerId == null) {
                throw new IllegalStateException("mirrorServerId is required");
            }
            if (mirrorServerName == null) {
                throw new IllegalStateException("mirrorServerName is required");
            }
            if (repository == null) {
                throw new IllegalStateException("repository is required");
            }
            if (source == null) {
                throw new IllegalStateException("source is required");
            }
        }
    }
}