package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.hook.repository.AbstractRepositoryHookRequest;
import com.atlassian.bitbucket.hook.repository.StandardRepositoryHookTrigger;
import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.repository.*;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;

import static com.atlassian.bitbucket.hook.repository.StandardRepositoryHookTrigger.FILE_EDIT;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Request that is provided to {@code repository-hook} plugin modules when a file is
 * {@link ContentService#editFile(EditFileRequest) edited}.
 *
 * @see ContentService#editFile(EditFileRequest)
 * @see StandardRepositoryHookTrigger
 * @since 5.0
 */
public class FileEditHookRequest extends AbstractRepositoryHookRequest {

    private static final String NULL_SHA1 = StringUtils.repeat("0", 40);

    private final Branch branch;
    private final InputSupplier<InputStream> content;
    private final String message;
    private final String path;
    private final RefChange refChange;
    private final String sourceCommitId;

    private FileEditHookRequest(Builder builder) {
        super(builder);

        branch = builder.branch;
        content = builder.content;
        message = builder.message;
        path = builder.path;
        sourceCommitId = builder.sourceCommitId;

        if (isDryRun() && builder.fromHash == null) {
            refChange = null;
        } else if (NULL_SHA1.equals(requireNonNull(builder.fromHash, "fromHash"))) {
            refChange = new SimpleRefChange.Builder()
                    .ref(branch)
                    .fromHash(NULL_SHA1)
                    .toHash(requireNonNull(builder.toHash, "toHash"))
                    .type(RefChangeType.ADD)
                    .build();
        } else {
            refChange = new SimpleRefChange.Builder()
                    .ref(branch)
                    .fromHash(builder.fromHash)
                    .toHash(requireNonNull(builder.toHash, "toHash"))
                    .type(RefChangeType.UPDATE)
                    .build();
        }
    }

    /**
     * @return the branch on which the edit operation would happen
     */
    @Nonnull
    public Branch getBranch() {
        return branch;
    }

    /**
     * @return a supplier providing access to the file content
     */
    @Nonnull
    public InputSupplier<InputStream> getContent() {
        return content;
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return refChange == null ? Collections.emptyList() : Collections.singleton(refChange);
    }

    /**
     * @return the commit message. Can be {@link Optional#empty()} if this is a {@link #isDryRun() dry-run} request.
     */
    @Nonnull
    public Optional<String> getMessage() {
        return ofNullable(message);
    }

    /**
     * @return the path of the edited or created file
     */
    @Nonnull
    public String getPath() {
        return path;
    }

    /**
    * @return the commit ID of the existing file in the repository, or {@code Optional#empty} if this is a new file
    */
    @Nonnull
    public Optional<String> getSourceCommitId() {
        return ofNullable(sourceCommitId);
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final Branch branch;
        private final InputSupplier<InputStream> content;
        private final String path;

        private String fromHash;
        private String toHash;
        private String message;
        private String sourceCommitId;

        public Builder(@Nonnull Repository repository, @Nonnull Branch branch,
                       @Nonnull String path, @Nonnull InputSupplier<InputStream> content) {
            super(repository, FILE_EDIT);

            this.branch = requireNonNull(branch, "branch");
            this.content = requireNonNull(content, "content");
            this.path = requireNonNull(path, "path");

            fromHash = toHash = branch.getLatestCommit();
        }

        @Nonnull
        public FileEditHookRequest build() {
            return new FileEditHookRequest(this);
        }

        @Nonnull
        public Builder fromHash(@Nullable String value) {
            fromHash = value;
            return self();
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            this.message = value;

            return self();
        }

        @Nonnull
        public Builder sourceCommitId(@Nullable String value) {
            this.sourceCommitId = value;

            return self();
        }

        @Nonnull
        public Builder toHash(@Nullable String value) {
            toHash = value;
            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
