package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.property.PropertySupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

/**
 * @see DiffContentCallback
 */
public interface Diff extends PropertySupport {

    /**
     * Retrieves the path for the changed file.
     * <p>
     * This value has three categories:
     * <ul>
     *     <li>The same as the {@link #getSource() source}, indicating the file was modified in-place</li>
     *     <li>Different from the {@link #getSource()} source}, indicating the file was copied, moved or
     *     renamed, potentially with or without content changes</li>
     *     <li>{@code null}, indicating the file was deleted</li>
     * </ul>
     * <p>
     * Note: While the destination and the {@link #getSource() source} may each be {@code null}, they will
     * never <i>both</i> be {@code null}.
     *
     * @return the path to the file after the change
     */
    @Nullable
    Path getDestination();

    /**
     * Retrieves a list containing zero or more {@link DiffHunk hunks} describing the changed lines
     * within the file. The list may be empty, but never {@code null}.
     *
     * @return zero or more hunks describing changed lines
     */
    @Nonnull
    List<DiffHunk> getHunks();

    /**
     * Retrieves the path for the file before it was changed.
     * <p>
     * This value has three categories:
     * <ul>
     *     <li>The same as the {@link #getDestination() destination}, indicating the file was modified in-place</li>
     *     <li>Different from the {@link #getDestination() destination}, indicating the file was copied, moved or
     *     renamed, potentially with or without content changes</li>
     *     <li>{@code null}, indicating the file is new</li>
     * </ul>
     * <p>
     * Note: While the {@link #getDestination() destination} and the source may each be {@code null}, they will never
     * <i>both</i> be {@code null}.
     *
     * @return the path to the file before the change
     */
    @Nullable
    Path getSource();

    /**
     * Retrieves a flag indicating whether the changed file is binary. Binary files cannot be diffed.
     * <p>
     * Note: When this flag is {@code true}, the {@link #getHunks() hunks} list will always be empty.
     *
     * @return {@code true} if the changed file is binary, indicating there is no diff; otherwise, {@code false}
     */
    boolean isBinary();

    /**
     * Retrieves a flag indicating whether the list of {@link #getHunks() hunks} had to be truncated.
     * <p>
     * To manage resource consumption, the system may impose a limit on the size of diffs it is willing to
     * render. This flag is set to inform callers when such limits are exceeded. Diffs may be truncated at
     * many different levels. Additional truncation flags exist for {@link DiffHunk#isTruncated() hunks},
     * {@link DiffSegment#isTruncated() segments} and {@link DiffLine#isTruncated() lines}, allowing callers
     * to know exactly how the diff was truncated.
     * <p>
     * Note: If this flag is {@code true} but no {@link DiffHunk hunks} are {@link DiffHunk#isTruncated()
     * truncated}, it means at least one hunk was <i>entirely</i> omitted.
     *
     * @return {@code true} if the diff was truncated, removing one or more
     */
    boolean isTruncated();
}
