package com.atlassian.bitbucket.content;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * Callback interface for streaming content tree nodes from a starting directory
 * <p>
 * Note: Implementors are <i>strongly</i> encouraged to extend from {@link AbstractContentTreeCallback}. This interface
 * <i>will</i> change, over time, and any class implementing it directly will be broken by such changes. Extending from
 * the abstract class will help prevent such breakages.
 *
 * @see AbstractContentTreeCallback
 */
public interface ContentTreeCallback {

    /**
     * Called after the final {@link #onTreeNode(ContentTreeNode) tree node} has been streamed.
     * <p>
     * Note: If the tree was empty, this method may be called immediately after {@link #onStart(ContentTreeContext)}
     * without any calls to {@link #onTreeNode(ContentTreeNode)}.
     *
     * @param summary summarizes the tree request and the streamed nodes
     * @throws IOException may be thrown by implementations which perform I/O.
     */
    void onEnd(@Nonnull ContentTreeSummary summary) throws IOException;

    /**
     * Called before the first {@link #onTreeNode(ContentTreeNode)}.
     *
     * @param context provides details about the tree request for which nodes are being streamed
     * @throws IOException may be thrown by implementations which perform I/O.
     */
    void onStart(@Nonnull ContentTreeContext context) throws IOException;

    /**
     * Called once for each path in the tree. {@link #onStart(ContentTreeContext)} is called before any invocation of this method.
     * {@link #onEnd(ContentTreeSummary) onEndPage(Page)} is called after all invocations of this method.
     *
     * @param node the current tree node
     * @return {@code true} if additional nodes should be provided; otherwise, {@code false} if the provided node should
     *         be the last
     * @throws IOException may be thrown by implementations which perform I/O operations, such as streaming tree nodes
     */
    boolean onTreeNode(@Nonnull ContentTreeNode node) throws IOException;
}
