package com.atlassian.bitbucket.user;

import com.atlassian.bitbucket.AuthorisationException;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;
import com.atlassian.bitbucket.validation.ArgumentValidationException;

import javax.annotation.Nonnull;

/**
 * Provides a history of repositories users have browsed.
 */
public interface RecentlyAccessedRepositoriesService {

    /**
     * Returns a page of repositories accessed most recently by the current user. The list is ordered most recent to
     * least recent. Only repositories the user {@link Permission#REPO_READ can currently access} are returned.
     * <p>
     * This method provides exactly the same result as {@link #findByCurrentUser(Permission, PageRequest)} invoked
     * with {@link Permission#REPO_READ}.
     *
     * @param pageRequest describes the page of recently accessed repositories to retrieve
     * @return page of repositories recently accessed by the current user
     * @throws AuthorisationException if there is no current effective user
     * @see #findByCurrentUser(Permission, PageRequest)
     */
    @Nonnull
    Page<Repository> findByCurrentUser(@Nonnull PageRequest pageRequest);

    /**
     * Returns a page of repositories accessed most recently by the current user. The list is ordered most recent to
     * least recent. Only repositories the user can currently access at the level specified by {@code permission}
     * are returned.
     *
     * @param permission    the permission that the effective user has to have with regards to the returned repositories.
     *                      The permission has to be {@link Permission#isResource()
     *                      repository-related}
     * @param pageRequest   describes the page of recently accessed repositories to retrieve
     * @return page of repositories recently accessed by the current user
     * @throws AuthorisationException if there is no current effective user
     * @throws ArgumentValidationException if the permission is not related to repository
     */
    @Nonnull
    Page<Repository> findByCurrentUser(@Nonnull Permission permission, @Nonnull PageRequest pageRequest);
}
