package com.atlassian.bitbucket.task;

import com.atlassian.bitbucket.comment.Comment;
import com.atlassian.bitbucket.comment.CommentService;
import com.atlassian.bitbucket.comment.CommentSeverity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Service to manage {@link Task}s.
 * <p>
 * As of 7.0, tasks have been deprecated and replaced with {@link CommentSeverity#BLOCKER} {@link Comment}s. As part of
 * that replacement, this service has been reimplemented as a facade for {@link CommentService}. App developers who use
 * tasks are <i>strongly</i> encouraged to update their code to use {@link CommentService} directly as soon as possible.
 * <p>
 * <b>Note</b>: {@link CommentSeverity#BLOCKER} comments have existed since 6.7, however, <i>prior to 7.0 they are not
 * treated as tasks.</i> Apps that wish to retain <i>semantic compatibility</i> with 6.x must continue using the
 * {@code TaskService} until support for pre-7.0 releases can be dropped.
 * <p>
 * Implementing tasks as {@code BLOCKER} comments imposes two <i>semantics-breaking</i> changes despite remaining
 * API-compatible with previous releases:
 * <ul>
 *     <li>Tasks can only be deleted by their authors, and repository administrators; pull request authors can
 *     no longer delete tasks they did not create (unless they're a repository administrator)</li>
 *     <li>Task text can only be updated by its author; pull request authors <i>and repository administrators</i>
 *     can no longer update the text for tasks they did not create</li>
 * </ul>
 * In addition to those semantics-breaking changes, there are other implications that may be worth noting:
 * <ul>
 *     <li>{@code BLOCKER} comments created using the {@link CommentService} directly are included in task results,
 *     <i>unless</i> they are root comments (i.e. they have no parent). It was not, and still is not, possible to
 *     create a task without an anchoring comment, so root {@code BLOCKER} comments cannot be treated as tasks because
 *     they would violate the nullability contract for {@link Task#getAnchor}</li>
 *     <li>{@code BLOCKER} comments created and/or updated using the task API still have their whitespace normalized</li>
 *     <li>Creating, deleting and updating tasks now raises comment events, which means those task operations can be
 *     canceled using the cancelable comment events</li>
 *     <li>Creating a new task creates a {@code BLOCKER} reply on the anchoring comment</li>
 *     <li>Deleting a task deletes the {@code BLOCKER} reply</li>
 *     <li>Updating a task updates the state and/or text of the {@code BLOCKER} reply</li>
 * </ul>
 *
 * @deprecated in 7.0 for removal in 8.0. Tasks are now managed using {@link Comment comments} with
 *             {@link CommentSeverity#BLOCKER BLOCKER} severity. App developers should use {@link CommentService} to
 *             create, retrieve and update {@code BLOCKER} comments; this service is now just a facade for that one.
 */
@Deprecated
public interface TaskService {

    /**
     * Create a task.
     * <p>
     * Since 7.0, this method creates a {@link CommentSeverity#BLOCKER BLOCKER} comment as a reply to the specified
     * {@link TaskCreateRequest#getAnchorId anchor comment}.
     *
     * @param request request detailing the task's properties
     * @return an instance of the created task
     */
    @Nonnull
    Task create(@Nonnull TaskCreateRequest request);

    /**
     * Delete a task.
     * <p>
     * Since 7.0, this method deletes the {@link CommentSeverity#BLOCKER BLOCKER} comment with the specified ID.
     * <p>
     * <b>Warning</b>: Since 7.0, the semantics of this method have changed: Pull request authors may no longer
     * delete tasks created by other users. As with comments, tasks can only be deleted by their authors and
     * repository administrators.
     *
     * @param taskId the ID of the task to delete
     */
    void delete(long taskId);

    /**
     * Get a task by its ID.
     * <p>
     * Since 7.0, this method retrieves the {@link CommentSeverity#BLOCKER BLOCKER} comment with the specified ID.
     * Due to the nullability contract of {@link Task#getAnchor}, if the {@code BLOCKER} comment is a root comment
     * (i.e. has no parent), it cannot be retrieved using this API.
     *
     * @param taskId the ID of the task to retrieve
     * @return the task matching the ID or {@code null} if none matches or the task is not visible to the current user
     */
    @Nullable
    Task getById(long taskId);

    /**
     * Update a task.
     * <p>
     * Since 7.0, this method updates the state and/or text of the {@link CommentSeverity#BLOCKER BLOCKER} comment.
     * <p>
     * <b>Warning</b>: Since 7.0, the semantics of this method have changed: Pull request authors and repository
     * administrators may no longer change the text of tasks created by other users. As with comments, task text
     * can only be changed by the task's author.
     *
     * @param request the properties to update
     * @return an instance of the updated task
     */
    @Nonnull
    Task update(@Nonnull TaskUpdateRequest request);

}
