package com.atlassian.bitbucket.scm.mirror;

import com.atlassian.bitbucket.scm.AbstractCommandParameters;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.io.File;
import java.nio.file.Path;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * @since 4.1
 */
public class MirrorSyncCommandParameters extends AbstractCommandParameters {

    private final boolean anonymous;
    private final Map<String, String> environment;
    private final boolean includePrivateRefs;
    private final String password;
    private final File privateKeyFile;
    private final boolean prune;
    private final Set<String> refspecs;
    private final String remoteUrl;
    private final String username;
    private final boolean withTags;

    private MirrorSyncCommandParameters(Builder builder) {
        environment = builder.environment.build();
        includePrivateRefs = builder.includePrivateRefs;
        password = builder.password;
        privateKeyFile = builder.privateKeyFile;
        prune = builder.prune;
        remoteUrl = builder.remoteUrl;
        refspecs = builder.refspecs.build();
        username = builder.username;
        withTags = builder.withTags;

        boolean credentialsSupplied = username != null || password != null || privateKeyFile != null;
        anonymous = builder.anonymous == null ? !credentialsSupplied : builder.anonymous;
        if (!anonymous && !credentialsSupplied) {
            throw new IllegalArgumentException("No credentials supplied, and 'anonymous' is not set");
        }
        if (anonymous && credentialsSupplied) {
            throw new IllegalArgumentException("Credentials supplied even though 'anonymous' is set");
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MirrorSyncCommandParameters that = (MirrorSyncCommandParameters) o;
        return anonymous == that.anonymous &&
               includePrivateRefs == that.includePrivateRefs &&
               prune == that.prune &&
               withTags == that.withTags &&
               Objects.equals(password, that.password) &&
               Objects.equals(privateKeyFile, that.privateKeyFile) &&
               Objects.equals(refspecs, that.refspecs) &&
               Objects.equals(remoteUrl, that.remoteUrl) &&
               Objects.equals(username, that.username);
    }

    @Override
    public int hashCode() {
        return Objects.hash(anonymous, includePrivateRefs, password, privateKeyFile, prune, refspecs, remoteUrl,
                username, withTags);
    }

    /**
     * @return a map of environment variables that will be applied to the synchronize command
     * @since 6.7
     */
    public Map<String, String> getEnvironment() {
        return environment;
    }

    /**
     * @return the password to provide for authentication
     * @since 4.6
     */
    @Nonnull
    public Optional<String> getPassword() {
        return ofNullable(password);
    }

    /**
     * @return the username to provide for authentication
     * @since 4.6
     */
    @Nonnull
    public Optional<String> getUsername() {
        return ofNullable(username);
    }

    /**
     * @return the SSH private key to use for authentication
     */
    @Nonnull
    public Optional<File> getPrivateKey() {
        return ofNullable(privateKeyFile);
    }

    /**
     * @return a set of refspecs to be passed to the fetch command, returns an empty set if not set
     * @since 6.7
     */
    @Nonnull
    public Set<String> getRefspecs() {
        return refspecs;
    }

    /**
     * @return the URL to fetch from
     */
    @Nonnull
    public String getRemoteUrl() {
        return remoteUrl;
    }

    /**
     * @return whether the sync command can be done in an anonymous context or not. If not provided, defaults to {@code false}
     * @since 6.10
     */
    public boolean isAnonymous() {
        return anonymous;
    }

    /**
     * If an explicit refspec is set then this setting is ignored.
     * @return true if private refs (outside refs/heads and refs/tags) should be included in the fetch
     * @since 5.1
     */
    public boolean isIncludePrivateRefs() {
        return includePrivateRefs;
    }

    /**
     * @return {@code true} when local branches and tags should be deleted where the upstream branch or tag has been deleted.
     * @since 6.7
     */
    public boolean isPrune() {
        return prune;
    }

    /**
     * If using a refspec that includes tags this must be set to false or the fetch command will fail.
     * @return true if tags should be included from fetch unless explicitly included in the refspec
     * @since 6.7
     */
    public boolean isWithTags() {
        return withTags;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("includePrivateRefs", includePrivateRefs)
                .add("password", password)
                .add("privateKeyFile", privateKeyFile)
                .add("prune", prune)
                .add("refspecs", refspecs)
                .add("remoteUrl", remoteUrl)
                .add("username", username)
                .add("withTags", withTags)
                .toString();
    }

    public static class Builder {

        private final String remoteUrl;

        private final ImmutableMap.Builder<String, String> environment;
        private final ImmutableSet.Builder<String> refspecs;
        private Boolean anonymous;
        private boolean includePrivateRefs;
        private String password;
        private File privateKeyFile;
        private boolean prune;
        private String username;
        private boolean withTags;

        /**
         * @param remoteUrl the URL to fetch from
         */
        public Builder(@Nonnull String remoteUrl) {
            this.remoteUrl = requireNonBlank(remoteUrl, "remoteUrl");
            environment = ImmutableMap.builder();
            includePrivateRefs = true;
            prune = true;
            refspecs = ImmutableSet.builder();
        }

        /**
         * @param mirrorSyncCommandParameters to copy
         * @since 6.7
         */
        public Builder(@Nonnull MirrorSyncCommandParameters mirrorSyncCommandParameters) {
            //Don't copy the anonymous flag because credentials might still be set on the builder and we want
            //anonymous to default to the correct value.
            environment = ImmutableMap.<String, String>builder().putAll(mirrorSyncCommandParameters.environment);
            remoteUrl = mirrorSyncCommandParameters.getRemoteUrl();
            includePrivateRefs = mirrorSyncCommandParameters.isIncludePrivateRefs();
            password = mirrorSyncCommandParameters.getPassword().orElse(null);
            privateKeyFile = mirrorSyncCommandParameters.getPrivateKey().orElse(null);
            prune = mirrorSyncCommandParameters.isPrune();
            username = mirrorSyncCommandParameters.getUsername().orElse(null);
            withTags = mirrorSyncCommandParameters.isWithTags();

            refspecs = ImmutableSet.builder();
            refspecs.addAll(mirrorSyncCommandParameters.getRefspecs());
        }

        /**
         * If no credentials are to be provided, set this to {@code true}. If no credentials are provided, and anonymous
         * is not set to {@code true}, then an {@link IllegalArgumentException} will be thrown.
         * Similarly, if credentials are provided, but anonymous is set to {@code true} then an
         * {@link IllegalArgumentException} will be thrown.
         *
         * @param value Whether or not the sync command should provide credentials
         * @return {@code this}
         * @since 6.10
         */
        @Nonnull
        public Builder anonymous(boolean value) {
            anonymous = value;
            return this;
        }

        @Nonnull
        public MirrorSyncCommandParameters build() {
            return new MirrorSyncCommandParameters(this);
        }

        /**
         * If an explicit refspec is set then this setting is ignored.
         * @param value whether to include private refs (outside refs/heads and refs/tags) in the fetch. The default
         *              value is true so all refs will be fetched.
         * @return {@code this}
         * @since 5.1
         */
        @Nonnull
        public Builder includePrivateRefs(boolean value) {
            includePrivateRefs = value;
            return this;
        }

        /**
         * If using a refspec that includes tags this must be set to false or the fetch command will fail.
         * @param value whether to include tags from the fetch unless included explicitly in a refspec.
         * @return the builder
         * @since 6.7
         */
        @Nonnull
        public Builder withTags(boolean value) {
            withTags = value;
            return this;
        }

        /**
         * @param value the password to use
         * @return {@code this}
         * @since 4.6
         */
        @Nonnull
        public Builder password(@Nonnull String value) {
            password = requireNonNull(value, "password");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return {@code this}
         */
        @Nonnull
        public Builder privateKey(@Nonnull File value) {
            privateKeyFile = requireNonNull(value, "privateKeyFile");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return {@code this}
         * @since 5.12
         */
        @Nonnull
        public Builder privateKey(@Nonnull Path value) {
            return privateKey(requireNonNull(value, "privateKeyFile").toFile());
        }

        /**
         * Specifies whether to supply {@code --prune} to prune local branches and tags where the upstream branch
         * or tag has been deleted.
         *
         * @param value true if pruning should be enabled
         * @return the builder
         * @since 6.7
         */
        @Nonnull
        public Builder prune(boolean value) {
            prune = value;
            return this;
        }

        /**
         * @param values refspecs to be passed to the fetch command
         * @return the builder
         * @since 6.7
         */
        @Nonnull
        public Builder refspecs(@Nonnull Iterable<String> values) {
            addIf(Objects::nonNull, refspecs, values);
            return this;
        }

        /**
         * @param value the username to use
         * @return {@code this}
         * @since 4.6
         */
        @Nonnull
        public Builder username(@Nonnull String value) {
            username = requireNonNull(value, "value");
            return this;
        }

        /**
         * Puts the provided {@code value} in the {@link #environment} map with the specified {@code name}, after ensuring
         * both the {@code name} and {@code value} are not blank.
         *
         * @param name  the name of the environment variable to set
         * @param value the value to set for the environment variable
         * @return the builder
         * @since 6.7
         */
        @Nonnull
        public Builder withEnvironment(@Nonnull String name, @Nonnull String value) {
            environment.put(requireNonBlank(name, "name"), requireNonBlank(value, "value"));
            return this;
        }
    }
}