package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

import static com.atlassian.bitbucket.pull.PullRequestMergeConfig.DEFAULT_COMMIT_SUMMARIES;
import static java.util.Objects.requireNonNull;

/**
 * @since 4.9
 */
public class SetPullRequestMergeConfigRequest extends AbstractPullRequestMergeConfigRequest {

    private final int commitSummaries;
    private final String defaultStrategyId;
    private final Set<String> enabledStrategyIds;

    private SetPullRequestMergeConfigRequest(Builder builder) {
        super(builder);

        commitSummaries = Math.max(builder.commitSummaries, 0); //Normalize negative to 0
        defaultStrategyId = builder.defaultStrategyId;
        enabledStrategyIds = builder.enabledStrategyIds.build();
    }

    /**
     * @return the number of commit summaries to append to the user-provided (or system-generated) message when
     *         merging a pull request, which may be {@code 0} to omit summaries
     * @since 6.7
     */
    public int getCommitSummaries() {
        return commitSummaries;
    }

    @Nonnull
    public String getDefaultStrategyId() {
        return defaultStrategyId;
    }

    @Nonnull
    public Set<String> getEnabledStrategyIds() {
        return enabledStrategyIds;
    }

    public static class Builder extends AbstractBuilder<Builder, SetPullRequestMergeConfigRequest> {

        private final ImmutableSet.Builder<String> enabledStrategyIds = ImmutableSet.builder();

        private int commitSummaries = DEFAULT_COMMIT_SUMMARIES;
        private String defaultStrategyId;

        public Builder(@Nonnull Project project, @Nonnull String scmId) {
            super(project, scmId);
        }

        public Builder(@Nonnull Repository repository) {
            super(repository);
        }

        public Builder(@Nonnull String scmId) {
            super(scmId);
        }

        @Nonnull
        @Override
        public SetPullRequestMergeConfigRequest build() {
            if (defaultStrategyId == null) {
                throw new IllegalStateException("The default strategy ID is required.");
            }

            return new SetPullRequestMergeConfigRequest(this);
        }

        /**
         * @param value the number of commit summaries to append to the user-supplied (or system-generated) message
         *              when merging a pull request, which may be {@code 0} to omit summaries
         * @return {@code this}
         * @since 6.7
         */
        @Nonnull
        public Builder commitSummaries(int value) {
            commitSummaries = value;

            return self();
        }

        @Nonnull
        public Builder defaultStrategyId(@Nonnull String value) {
            defaultStrategyId = requireNonBlank(value, "defaultStrategyId");

            return self();
        }

        @Nonnull
        public Builder enabledStrategyId(@Nonnull String value) {
            enabledStrategyIds.add(requireNonBlank(value, "enabledStrategyId"));

            return self();
        }

        @Nonnull
        public Builder enabledStrategyIds(@Nonnull Iterable<String> values) {
            requireNonNull(values, "enabledStrategyIds");
            values.forEach(this::enabledStrategyId);

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
